<?php

namespace RAP;

/**
 * All public methods return a redirect URL.
 */
class LoginHandler {

    protected $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    public function onIdentityDataReceived(Identity $identity): string {

        $session = $this->locator->getSession();
        $session->setLoginIdentityType($identity->type);

        $user = $this->getUser($identity);

        $joinStep = $this->checkJoinAction($session, $user);
        if ($joinStep !== null) {
            return $joinStep;
        }

        $session->setUser($user);

        $autoJoinStep = $this->checkAutoJoin();
        if ($autoJoinStep !== null) {
            return $autoJoinStep;
        }

        if ($user->id === null) {
            return $this->redirectToTOUCheck($user);
        } else {
            $this->locator->getUserHandler()->updateIdentity($user, $identity);
        }

        return $this->getAfterLoginRedirect();
    }

    private function getUser(Identity $identity): User {

        $userDao = $this->locator->getUserDAO();

        $user = $userDao->findUserByIdentity($identity->type, $identity->typedId);

        if ($user === null) {
            // create new user with null id
            $user = new User();
            $user->addIdentity($identity);
        }

        return $user;
    }

    private function checkJoinAction(SessionData $session, User $user): ?string {
        if ($session->getOAuth2RequestData() === null && $session->getAction() === 'join' && $session->getUser() !== null) {
            if ($session->getUser()->id === $user->id) {
                // attempting to join an already joined user: go to account page without joining
                $session->setAction('account');
            } else {
                return $this->showConfirmJoin($user);
            }
        }
        return null;
    }

    public function confirmJoin(): string {

        $session = $this->locator->getSession();

        if ($session->getUserToJoin() === null) {
            throw new BadRequestException("Unable to find user to join");
        }

        $this->joinUsers();

        return $this->getAfterLoginRedirect();
    }

    private function checkAutoJoin(): ?string {

        $userDao = $this->locator->getUserDAO();
        $session = $this->locator->getSession();
        $user = $session->getUser();

        if ($user->id === null) {
            $joinableUsers = $userDao->findJoinableUsersByEmail($user->identities[0]->email);
        } else {
            $joinableUsers = $userDao->findJoinableUsersByUserId($user->id);
        }

        if (count($session->getRejectedJoins()) > 0) {
            $joinableUsers = array_values(array_diff($joinableUsers, $session->getRejectedJoins()));
        }

        if (count($joinableUsers) > 0) {
            // select first user
            $userToJoin = $userDao->findUserById($joinableUsers[0]);
            $session->setAutojoin(true);
            return $this->showConfirmJoin($userToJoin);
        }

        $session->setAutojoin(false);
        return null;
    }

    public function rejectJoin(): string {

        $session = $this->locator->getSession();

        $user = $session->getUser();
        if ($user === null) {
            throw new \RAP\BadRequestException("Unable to find user");
        }

        if ($user->id === null) {
            $session->addRejectedJoin($session->getUserToJoin()->id);
        } else {
            $this->locator->getUserDAO()
                    ->insertRejectedJoin($user->id, $session->getUserToJoin()->id);
        }

        return $this->getAfterLoginRedirect();
    }

    private function showConfirmJoin(User $userToJoin): string {
        $this->locator->getSession()->setUserToJoin($userToJoin);
        return $this->locator->getBasePath() . '/confirm-join';
    }

    /**
     * Stores the data into session and Redirect to Term of Use acceptance page.
     */
    private function redirectToTOUCheck(): string {
        return $this->locator->getBasePath() . '/tou-check';
    }

    /**
     * Stores the user data into the database after he/she accepted the Terms of Use.
     */
    public function register(): string {

        $session = $this->locator->getSession();
        $user = $session->getUser();

        if ($user === null) {
            throw new BadRequestException("User data not retrieved.");
        } else {

            $this->locator->getUserHandler()->saveUser($user);

            // save rejected joins stored in session
            foreach ($session->getRejectedJoins() as $userId) {
                $this->locator->getUserDAO()->insertRejectedJoin($user->id, $userId);
            }

            return $this->getAfterLoginRedirect();
        }
    }

    private function joinUsers(): void {

        $session = $this->locator->getSession();
        $user = $session->getUser();
        $userToJoin = $session->getUserToJoin();

        $joinedUser = $this->locator->getUserHandler()->joinUsers($user, $userToJoin);
        $session->setUser($joinedUser);

        if ($session->getAction() === 'join') {
            $session->setAction('account');
        }

        $session->setUserToJoin(null);
    }

    private function getAfterLoginRedirect(): string {

        $autoJoinStep = $this->checkAutoJoin();
        if ($autoJoinStep !== null) {
            return $autoJoinStep;
        }

        $session = $this->locator->getSession();
        $user = $session->getUser();

        if ($user->id === null) {
            return $this->redirectToTOUCheck($user);
        }

        $this->locator->getAuditLogger()->info("LOGIN," . $session->getLoginIdentityType() . "," . $user->id);

        if ($session->getOAuth2RequestData() !== null) {
            // Redirect to OAuth2 client callback URL
            $redirectUrl = $this->locator->getOAuth2RequestHandler()->getRedirectResponseUrl();
            session_destroy();
            return $redirectUrl;
        }

        $action = $session->getAction();

        if ($action === 'account' || $action === 'admin') {
            return $this->locator->getBasePath() . '/' . $action;
        }

        throw new \Exception("Unable to find a proper redirect");
    }

}
