<?php

/*
 * This file is part of rap
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

use PHPUnit\Framework\TestCase;
use \Firebase\JWT\JWT;

final class TokenBuilderTest extends TestCase {

    public function testJWTCreation() {

        $jwksDAOStub = $this->createMock(\RAP\JWKSDAO::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getJWKSDAO')->willReturn($jwksDAOStub);

        $jwksHandler = new \RAP\JWKSHandler($locatorStub);
        $keyPair = $jwksHandler->generateKeyPair();

        $jwksDAOStub->method('getNewestKeyPair')->willReturn($keyPair);

        $user = $this->getUser();

        $daoStub = $this->createMock(\RAP\UserDAO::class);
        $locatorStub->method('getUserDAO')->willReturn($daoStub);
        $daoStub->method('findUserById')->willReturn($user);

        $locatorStub->config = json_decode('{"jwtIssuer": "issuer"}');

        $accessToken = new \RAP\AccessTokenData();
        $accessToken->token = "ttt";
        $accessToken->scope = ["email", "profile"];
        $accessToken->userId = "user_id";

        $tokenBuilder = new \RAP\TokenBuilder($locatorStub);
        $jwt = $tokenBuilder->getIdToken($accessToken);

        $this->assertNotNull($jwt);

        $payload = JWT::decode($jwt, $keyPair->publicKey, [$keyPair->alg]);

        $this->assertEquals("issuer", $payload->iss);
        $this->assertEquals($user->id, $payload->sub);
        $this->assertEquals($user->getCompleteName(), $payload->name);
        $identity = $user->identities[0];
        $this->assertEquals($identity->name, $payload->given_name);
        $this->assertEquals($identity->surname, $payload->family_name);
        $this->assertEquals($identity->institution, $payload->org);
    }

    public function testGetAccessToken(): void {

        $jwksDAOStub = $this->createMock(\RAP\JWKSDAO::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getJWKSDAO')->willReturn($jwksDAOStub);

        $jwksHandler = new \RAP\JWKSHandler($locatorStub);
        $keyPair = $jwksHandler->generateKeyPair();

        $jwksDAOStub->method('getNewestKeyPair')->willReturn($keyPair);

        $userDaoStub = $this->createMock(\RAP\UserDAO::class);
        $locatorStub->method('getUserDAO')->willReturn($userDaoStub);

        $user = $this->getUser();
        $userDaoStub->method('findUserById')->willReturn($user);

        $client = new \RAP\BrowserBasedOAuth2Client((object) [
                    "id" => "aao",
                    "secret" => "2a97516c354b68848cdbd8f54a226a0a55b21ed138e207ad6c5cbb9c00aa5aea",
                    "redirect" => "redirect",
                    "scope" => "openid email",
                    "methods" => [],
                    "scopeAudienceMap" => ["read:gms" => "gms"]
        ]);

        $locatorStub->method('getBrowserBasedOAuth2ClientById')->willReturn($client);

        $locatorStub->config = json_decode('{"jwtIssuer": "issuer"}');

        $tokenBuilder = new \RAP\TokenBuilder($locatorStub);

        $tokenData = new \RAP\AccessTokenData();
        $tokenData->token = "ttt";
        $tokenData->scope = ["openid", "read:gms"];
        $tokenData->userId = "user_id";
        $tokenData->clientId = "aao";

        $jwt = $tokenBuilder->getAccessToken($tokenData);

        $this->assertNotNull($jwt);
    }

    private function getUser(): \RAP\User {
        $user = new \RAP\User();
        $user->id = "user_id";
        $identity = new \RAP\Identity(\RAP\Identity::EDU_GAIN);
        $identity->email = "name@inaf.it";
        $identity->name = "Name";
        $identity->surname = "Surname";
        $identity->primary = true;
        $identity->institution = "INAF";
        $user->addIdentity($identity);
        return $user;
    }

}
