<?php

/*
 * This file is part of rap
 * Copyright (C) 2020 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

namespace RAP;

/**
 * See https://tools.ietf.org/html/rfc8693
 */
class TokenExchanger {

    private $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    public function exchangeToken(array $params, array $headers): array {

        $this->locator->getClientAuthChecker()->validateClientAuth($headers);

        if ($params['subject_token'] === null) {
            throw new BadRequestException("subject_token is required");
        }
        if ($params['subject_token_type'] === null) {
            throw new BadRequestException("subject_token_type is required");
        }
        if (strtolower($params['subject_token_type']) !== 'bearer') {
            throw new BadRequestException("subject_token_type " . $params['subject_token_type'] . " not supported");
        }

        $subjectToken = $this->locator->getTokenChecker()->getValidTokenObject($params['subject_token']);

        $claims = array(
            'sub' => $subjectToken->sub
        );

        if ($params['resource'] !== null) {
            $claims['resource'] = $params['resource'];
            $claims['jti'] = uniqid();
        }
        if ($params['audience'] !== null) {
            $claims['aud'] = $this->getAudienceClaim($params['audience']);
        }
        if ($params['scope'] !== null) {
            $claims['scope'] = $params['scope'];
        }
        if ($params['expires_in'] !== null) {
            $claims['exp'] = time() + intval($params['expires_in']);
        }

        $accessToken = $this->locator->getTokenBuilder()->generateToken($claims);

        $data = [];

        $data['access_token'] = $accessToken;
        $data['issued_token_type'] = "urn:ietf:params:oauth:token-type:jwt";
        $data['token_type'] = 'Bearer';
        $data['expires_in'] = $params['expires_in'] !== null ? $params['expires_in'] : 3600;

        return $data;
    }

    private function getAudienceClaim($audienceParam) {
        $audiences = explode(' ', $audienceParam);
        if (count($audiences) === 1) {
            // according to RFC 7519 audience can be a single value or an array
            return $audiences[0];
        }
        return $audiences;
    }

}
