<?php

namespace RAP;

class LoginHandler {

    protected $locator;
    private $identityType;

    public function __construct(Locator $locator, string $identityType) {
        $this->locator = $locator;
        $this->identityType = $identityType;
        $this->locator->getSession()->setLoginIdentityType($identityType);
    }

    public function onIdentityDataReceived(string $typedId, \Closure $fillIdentityData): string {

        $user = $this->locator->getUserDAO()->findUserByIdentity($this->identityType, $typedId);

        if ($user === null) {
            return $this->handleNewIdentity($typedId, $fillIdentityData);
        } else {
            $this->updateUser($user, $typedId, $fillIdentityData);
        }

        $session = $this->locator->getSession();
        if ($session->getOAuth2RequestData() === null && $session->getAction() === 'join' &&
                $session->getUser() !== null && $session->getUser()->id !== $user->id) {
            return $this->showConfirmJoin($user);
        }

        return $this->getAfterLoginRedirect($user);
    }

    protected function handleNewIdentity(string $typedId, \Closure $fillIdentityData): string {

        $session = $this->locator->getSession();

        if ($session->getUser() !== null && $session->getAction() === 'join') {
            $userToJoin = $this->getNewUser($typedId, $fillIdentityData);
            return $this->showConfirmJoin($userToJoin);
        } else {
            return $this->redirectToTOUCheck($typedId, $fillIdentityData);
        }
    }

    private function showConfirmJoin(User $userToJoin): string {
        $this->locator->getSession()->setUserToJoin($userToJoin);
        return $this->locator->getBasePath() . '/confirm-join';
    }

    /**
     * Stores the data into session and Redirect to Term of Use acceptance page.
     */
    private function redirectToTOUCheck(string $typedId, \Closure $fillIdentityData): string {

        // Create new user
        $user = $this->getNewUser($typedId, $fillIdentityData);

        $this->locator->getSession()->setUser($user);

        return $this->locator->getBasePath() . '/tou-check';
    }

    private function getNewUser(string $typedId, \Closure $fillIdentityData): User {
        $user = new User();
        $identity = new Identity($this->identityType);
        $identity->typedId = $typedId;
        $fillIdentityData($identity);
        $user->addIdentity($identity);
        return $user;
    }

    /**
     * Update user with fresh information received by IdP. Useful for keeping email address always updated.
     */
    private function updateUser(User $user, string $typedId, \Closure $fillIdentityData): void {
        $identity = $user->getIdentityByTypedId($typedId);
        $fillIdentityData($identity);
        $this->locator->getUserDAO()->updateIdentity($identity);
    }

    public function getAfterLoginRedirect(User $user): string {

        $session = $this->locator->getSession();
        $this->locator->getAuditLogger()->info("LOGIN," . $this->identityType . "," . $user->id);

        if ($session->getOAuth2RequestData() !== null) {
            $session->setUser($user);
            $redirectUrl = $this->locator->getOAuth2RequestHandler()->getRedirectResponseUrl();
            session_destroy();
            return $redirectUrl;
        }

        if ($session->getAction() !== null) {

            $action = $session->getAction();

            if ($action === 'join') {
                $user = $this->joinTo($user);
                $action = 'account';
                $session->setAction($action);
            }

            $session->setUser($user);

            if ($action === 'account' || $action === 'admin') {
                return $this->locator->getBasePath() . '/' . $action;
            }
        }

        throw new \Exception("Unable to find a proper redirect");
    }

    private function joinTo(User $userToJoin): User {

        $session = $this->locator->getSession();
        $user = $session->getUser();

        if ($user === null) {
            return $userToJoin;
        }

        if ($userToJoin->id === null) {
            // New identity, not yet associated with an user: simply add it to
            // previously logged in user.
            $identity = $userToJoin->identities[0];
            $user->addIdentity($identity);
            $this->locator->getUserHandler()->saveUser($user);
        } else if ($user->id !== $userToJoin->id) {
            $user = $this->locator->getUserHandler()->joinUsers($user, $userToJoin);
        }

        $session->setUserToJoin(null);

        return $user;
    }

}
