<?php

namespace RAP;

class OrcidLogin extends LoginHandler {

    public function __construct(Locator $locator) {
        parent::__construct($locator);
    }

    public function login(): string {

        $ORCID = $this->locator->config->authenticationMethods->OrcID;

        $url = "https://orcid.org/oauth/authorize";
        $url = $url . "?client_id=" . $ORCID->id;
        $url = $url . "&response_type=code";
        $url = $url . "&scope=/authenticate";
        $url = $url . "&redirect_uri=" . $this->locator->getBasePath() . $ORCID->callback;

        return $url;
    }

    public function retrieveToken($code) {

        if ($code === null) {
            throw new BadRequestException("Unable to get ORCID client code");
        }

        $token = $this->getAccessTokenFromCode($code);

        $access_token = $token['access_token'];
        $expires_in = $token['expires_in'];
        $orcid_id = $username = $token['orcid'];

        $orcid_data = $this->getOrcidDataUsingAccessToken($orcid_id, $access_token);

        if (!isset($orcid_data['person']['emails']['email'][0]['email'])) {
            throw new \Exception("ORCID didn't return the email");
        }

        $identity = new Identity(Identity::ORCID);
        $identity->typedId = $orcid_id;
        $identity->email = $email = $orcid_data['person']['emails']['email'][0]['email'];
        $identity->name = $orcid_data['person']['name']['given-names']['value'];
        $identity->surname = $orcid_data['person']['name']['family-name']['value'];
        $employmentSummary = $orcid_data['activities-summary']['employments']['employment-summary'];
        if (count($employmentSummary) > 0) {
            $identity->institution = $employmentSummary[0]['organization']['name'];
        }

        return $this->onIdentityDataReceived($identity);
    }

    private function getAccessTokenFromCode($code): array {

        $ORCID = $this->locator->config->authenticationMethods->OrcID;

        //create array of data to be posted to get AccessToken
        $post_data = array(
            'grant_type' => "authorization_code",
            'code' => $code,
            'redirect_uri' => $this->locator->getBasePath() . $ORCID->callback,
            'client_id' => $ORCID->id,
            'client_secret' => $ORCID->secret
        );

        //traverse array and prepare data for posting (key1=value1)
        foreach ($post_data as $key => $value) {
            $post_items[] = $key . '=' . $value;
        }

        //create the final string to be posted
        $post_string = implode('&', $post_items);

        //create cURL connection
        $conn = curl_init('https://orcid.org/oauth/token');

        //set options
        curl_setopt($conn, CURLOPT_CONNECTTIMEOUT, 30);
        curl_setopt($conn, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($conn, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($conn, CURLOPT_FOLLOWLOCATION, 1);

        //set data to be posted
        curl_setopt($conn, CURLOPT_POSTFIELDS, $post_string);

        //perform our request
        $result = curl_exec($conn);

        if ($result) {
            $token = json_decode($result, TRUE);
            curl_close($conn);
            return $token;
        } else {
            //show information regarding the error
            $errorMessage = curl_errno($conn) . "-";
            $errorMessage = $errorMessage . curl_error($conn);
            curl_close($conn);
            throw new \Exception($errorMessage);
        }
    }

    private function getOrcidDataUsingAccessToken(string $orcid_id, string $access_token) {

        // API call
        $orcid_url = "https://pub.orcid.org/v2.1/" . $orcid_id . "/record";
        $conn = curl_init();
        curl_setopt($conn, CURLOPT_URL, $orcid_url);
        curl_setopt($conn, CURLOPT_HTTPHEADER, array(
            'Authorization: Bearer ' . $access_token,
            'Accept: application/json'));

        curl_setopt($conn, CURLOPT_RETURNTRANSFER, true);
        $result = curl_exec($conn);

        if ($result) {
            $orcid_data = json_decode($result, TRUE);
            curl_close($conn);
            return $orcid_data;
        } else {
            $errorMessage = curl_errno($conn) . "-";
            $errorMessage = $errorMessage . curl_error($conn);
            curl_close($conn);
            throw new \Exception($errorMessage);
        }
    }

}
