<?php

namespace RAP;

/**
 * Class implementing the locator pattern in order to implement a rough dependency injection.
 */
class Locator {

    use ClientsLocator;

    public $config;
    private $serviceLogger;
    private $auditLogger;
    private $session;
    private $version;

    public function __construct($config) {
        $this->config = $config;

        $this->setupLoggers();
        $this->version = file_get_contents(ROOT . '/version.txt');
    }

    public function getVersion(): string {
        return $this->version;
    }

    public function getProtocol(): string {
        return (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https://' : 'http://';
    }

    public function getBasePath(): string {
        return $this->getProtocol() . $_SERVER['HTTP_HOST'] . $this->config->contextRoot;
    }

    public function getUserDAO(): UserDAO {
        $databaseConfig = $this->config->databaseConfig;
        switch ($databaseConfig->dbtype) {
            case 'MySQL':
                return new MySQLUserDAO($this);
            default:
                throw new \Exception($databaseConfig->dbtype . ' not supported yet');
        }
    }

    public function getJWKSDAO(): JWKSDAO {
        $databaseConfig = $this->config->databaseConfig;
        switch ($databaseConfig->dbtype) {
            case 'MySQL':
                return new MySQLJWKSDAO($this);
            default:
                throw new \Exception($databaseConfig->dbtype . ' not supported yet');
        }
    }

    public function getAccessTokenDAO(): AccessTokenDAO {
        $databaseConfig = $this->config->databaseConfig;
        switch ($databaseConfig->dbtype) {
            case 'MySQL':
                return new MySQLAccessTokenDAO($this);
            default:
                throw new \Exception($databaseConfig->dbtype . ' not supported yet');
        }
    }

    public function getRefreshTokenDAO(): RefreshTokenDAO {
        $databaseConfig = $this->config->databaseConfig;
        switch ($databaseConfig->dbtype) {
            case 'MySQL':
                return new MySQLRefreshTokenDAO($this);
            default:
                throw new \Exception($databaseConfig->dbtype . ' not supported yet');
        }
    }

    public function getUserHandler(): UserHandler {
        return new UserHandler($this);
    }

    public function getOAuth2RequestHandler(): OAuth2RequestHandler {
        return new OAuth2RequestHandler($this);
    }

    public function getTokenBuilder(): TokenBuilder {
        return new TokenBuilder($this);
    }

    public function getTokenChecker(): TokenChecker {
        return new TokenChecker($this);
    }

    public function getClientAuthChecker(): ClientAuthChecker {
        return new ClientAuthChecker($this);
    }

    public function getTokenExchanger(): TokenExchanger {
        return new TokenExchanger($this);
    }

    public function getGmsClient(): GmsClient {
        return new GmsClient($this);
    }

    /**
     * Retrieve the SessionData object from the $_SESSION PHP variable. Create a
     * new one if it is necessary.
     */
    public function getSession(): SessionData {
        if (isset($_SESSION[\RAP\SessionData::KEY])) {
            $this->session = $_SESSION[SessionData::KEY];
        } else {
            $this->session = new \RAP\SessionData();
            $this->session->save();
        }
        return $this->session;
    }

    public function getServiceLogger(): \Monolog\Logger {
        return $this->serviceLogger;
    }

    public function getAuditLogger(): \Monolog\Logger {
        return $this->auditLogger;
    }

    public function getJWKSHandler(): JWKSHandler {
        return new JWKSHandler($this);
    }

    private function setupLoggers() {
        // Monolog require timezone to be set
        date_default_timezone_set($this->config->timeZone);

        $logLevel = array_search($this->config->logLevel, \Monolog\Logger::getLevels());

        $this->serviceLogger = new \Monolog\Logger('serviceLogger');
        $this->serviceLogger->pushHandler(new \Monolog\Handler\StreamHandler($this->config->serviceLogFile, $logLevel));

        $this->auditLogger = new \Monolog\Logger('auditLogger');
        $this->auditLogger->pushHandler(new \Monolog\Handler\StreamHandler($this->config->auditLogFile, $logLevel));
    }

}
