<?php

use PHPUnit\Framework\TestCase;

final class OAuth2RequestHandlerTest extends TestCase {

    public function testBadRequestExceptionIfMissingClientId(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => null
        ];

        $locatorStub = $this->createMock(\RAP\Locator::class);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testInvalidClientRedirectURI(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "invalid_redirect_uri",
            "state" => "state",
            "alg" => null,
            "nonce" => null,
            "scope" => "email%20profile"
        ];

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getBrowserBasedOAuth2ClientById')->willReturn(new \RAP\BrowserBasedOAuth2Client((object) [
                            "id" => "client_id",
                            "secret" => hash('sha256', "foo"),
                            "redirect" => "redirect_uri",
                            "scope" => "email profile",
                            "methods" => []
        ]));

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testExecuteOAuthStateFlow(): void {

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "redirect_uri",
            "state" => "state",
            "alg" => null,
            "nonce" => null,
            "scope" => "email%20profile"
        ];

        $client = new \RAP\BrowserBasedOAuth2Client((object) [
                    "id" => "client_id",
                    "secret" => hash('sha256', "foo"),
                    "redirect" => "redirect_uri",
                    "scope" => "email profile",
                    "methods" => []
        ]);
        $client->redirectUrl = "redirect_uri";

        $sessionStub = $this->createMock(\RAP\SessionData::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getBrowserBasedOAuth2ClientById')->willReturn($client);
        $locatorStub->method('getSession')->willReturn($sessionStub);

        $sessionStub->expects($this->once())
                ->method('setOAuth2RequestData')->with($this->anything());

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testHandleCheckTokenRequest(): void {

        $tokenBuilderStub = $this->createMock(\RAP\TokenBuilder::class);
        $tokenBuilderStub->method('getIdToken')->willReturn('id-token');

        $tokenCheckerStub = $this->createMock(\RAP\TokenChecker::class);
        $tokenCheckerStub->method('validateToken')->willReturn((object) [
                    "sub" => "123",
                    "iat" => time(),
                    "exp" => time() + 3600,
                    "scope" => "openid email",
                    "aud" => "my-client"
        ]);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getTokenChecker')->willReturn($tokenCheckerStub);
        $locatorStub->method('getTokenBuilder')->willReturn($tokenBuilderStub);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $result = $requestHandler->handleCheckTokenRequest(['Authorization' => 'Bearer: <token>']);

        $this->assertEquals(3600, $result['exp']);
        $this->assertEquals('123', $result['user_name']);
        $this->assertEquals('my-client', $result['client_id']);
        $this->assertEquals('id-token', $result['id_token']);
    }

    public function testHandleGetTokenFromCodeRequest(): void {

        $authCheckerStub = $this->createMock(\RAP\ClientAuthChecker::class);

        $tokenData = new \RAP\AccessTokenData();
        $tokenData->redirectUri = "redirect";
        $tokenData->scope = ['openid'];

        $tokenDaoStub = $this->createMock(\RAP\AccessTokenDAO::class);
        $tokenDaoStub->method('retrieveTokenDataFromCode')->willReturn($tokenData);

        $tokenBuilderStub = $this->createMock(\RAP\TokenBuilder::class);
        $tokenBuilderStub->method('getAccessToken')->willReturn('<access_token>');

        $refreshTokenDaoStub = $this->createMock(\RAP\RefreshTokenDAO::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getClientAuthChecker')->willReturn($authCheckerStub);
        $locatorStub->method('getAccessTokenDAO')->willReturn($tokenDaoStub);
        $locatorStub->method('getTokenBuilder')->willReturn($tokenBuilderStub);
        $locatorStub->method('getRefreshTokenDAO')->willReturn($refreshTokenDaoStub);

        $authCheckerStub->expects($this->once())
                ->method('validateClientAuth')->with($this->anything());

        $tokenDaoStub->expects($this->once())
                ->method('retrieveTokenDataFromCode')->with($this->anything());

        $tokenDaoStub->expects($this->once())
                ->method('deleteTokenData')->with($this->anything());

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $params = [
            "grant_type" => "authorization_code",
            "redirect_uri" => "redirect",
            "code" => "123"
        ];
        $result = $requestHandler->handleAccessTokenRequest($params, []);

        $this->assertEquals(3600, $result['expires_in']);
        $this->assertEquals("Bearer", $result['token_type']);
        $this->assertEquals("<access_token>", $result['access_token']);
        $this->assertNotNull($result['refresh_token']);
    }

    public function testHandleClientCredentialsRequest(): void {

        $authCheckerStub = $this->createMock(\RAP\ClientAuthChecker::class);

        $tokenBuilderStub = $this->createMock(\RAP\TokenBuilder::class);
        $tokenBuilderStub->method('getAccessToken')->willReturn('<access_token>');

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getClientAuthChecker')->willReturn($authCheckerStub);
        $locatorStub->method('getTokenBuilder')->willReturn($tokenBuilderStub);

        $authCheckerStub->expects($this->once())
                ->method('validateCliClientAuth')->with($this->anything());

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $params = [
            "grant_type" => "client_credentials"
        ];
        $result = $requestHandler->handleAccessTokenRequest($params, []);

        $this->assertEquals(3600, $result['expires_in']);
        $this->assertEquals("Bearer", $result['token_type']);
        $this->assertEquals("<access_token>", $result['access_token']);
        $this->assertFalse(isset($result['refresh_token']));
    }

    public function testHandleRefreshTokenRequest(): void {

        $refreshTokenData = new \RAP\RefreshTokenData();
        $refreshTokenData->scope = ['openid', 'email'];

        $authCheckerStub = $this->createMock(\RAP\ClientAuthChecker::class);

        $tokenBuilderStub = $this->createMock(\RAP\TokenBuilder::class);
        $tokenBuilderStub->method('getAccessToken')->willReturn('<access_token>');

        $refreshTokenDaoStub = $this->createMock(\RAP\RefreshTokenDAO::class);
        $refreshTokenDaoStub->method('getRefreshTokenData')->willReturn($refreshTokenData);

        $tokenDaoStub = $this->createMock(\RAP\AccessTokenDAO::class);
        $tokenDaoStub->method('createTokenData')->will($this->returnArgument(0));

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getClientAuthChecker')->willReturn($authCheckerStub);
        $locatorStub->method('getTokenBuilder')->willReturn($tokenBuilderStub);
        $locatorStub->method('getRefreshTokenDAO')->willReturn($refreshTokenDaoStub);
        $locatorStub->method('getAccessTokenDAO')->willReturn($tokenDaoStub);

        $authCheckerStub->expects($this->once())
                ->method('validateClientAuth')->with($this->anything());

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $params = [
            "grant_type" => "refresh_token",
            "refresh_token" => "<refresh_token>",
            "scope" => "openid email"
        ];
        $result = $requestHandler->handleAccessTokenRequest($params, []);

        $this->assertEquals(3600, $result['expires_in']);
        $this->assertEquals("Bearer", $result['token_type']);
        $this->assertEquals("<access_token>", $result['access_token']);
        $this->assertNotNull($result['refresh_token']);
    }

    public function testGetRedirectResponseUrlForAuthorizationCodeFlow(): void {

        $user = new \RAP\User();
        $user->id = "123";

        $requestData = new \RAP\OAuth2RequestData();
        $requestData->clientId = "<client-id>";
        $requestData->redirectUrl = "<base-path>";
        $requestData->scope = ["openid", "profile"];
        $requestData->state = "<state>";

        $sessionStub = $this->createMock(\RAP\SessionData::class);
        $sessionStub->method('getUser')->willReturn($user);
        $sessionStub->method('getOAuth2RequestData')->willReturn($requestData);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getSession')->willReturn($sessionStub);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $result = $requestHandler->getRedirectResponseUrl();

        $url_components = parse_url($result);
        $this->assertEquals('<base-path>', $url_components['path']);

        $params = [];
        parse_str($url_components['query'], $params);

        $this->assertEquals("<state>", $params['state']);
        $this->assertEquals("openid profile", $params['scope']);

        $this->assertNotNull($params['code']);
    }

    public function testGetRedirectResponseUrlForImplicitFlow(): void {

        $user = new \RAP\User();
        $user->id = "123";

        $requestData = new \RAP\OAuth2RequestData();
        $requestData->clientId = "<client-id>";
        $requestData->redirectUrl = "<base-path>";
        $requestData->scope = ["openid", "profile"];

        $sessionStub = $this->createMock(\RAP\SessionData::class);
        $sessionStub->method('getUser')->willReturn($user);
        $sessionStub->method('getOAuth2RequestData')->willReturn($requestData);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getSession')->willReturn($sessionStub);

        $tokenBuilderStub = $this->createMock(\RAP\TokenBuilder::class);
        $tokenBuilderStub->method('getIdToken')->willReturn('<id_token>');
        $locatorStub->method('getTokenBuilder')->willReturn($tokenBuilderStub);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $result = $requestHandler->getRedirectResponseUrl();

        $this->assertEquals("<base-path>#id_token=<id_token>", $result);
    }

}
