<?php

use PHPUnit\Framework\TestCase;

session_start();

final class LoginHandlerTest extends TestCase {

    private $locatorStub;
    private $userDaoStub;
    private $sessionStub;
    private $oAuth2RequestHandler;
    private $auditLogger;
    private $loginHandler;

    public function setUp(): void {
        $this->locatorStub = $this->createMock(\RAP\Locator::class);
        $this->locatorStub->method('getBasePath')->willReturn('http://rap-ia2');

        $this->userDaoStub = $this->createMock(\RAP\UserDAO::class);
        $this->locatorStub->method('getUserDAO')->willReturn($this->userDaoStub);

        $this->sessionStub = $this->createMock(\RAP\SessionData::class);
        $this->locatorStub->method('getSession')->willReturn($this->sessionStub);

        $this->oAuth2RequestHandler = $this->createMock(\RAP\OAuth2RequestHandler::class);
        $this->locatorStub->method('getOAuth2RequestHandler')->willReturn($this->oAuth2RequestHandler);

        $this->auditLogger = $this->createMock(\Monolog\Logger::class);
        $this->locatorStub->method('getAuditLogger')->willReturn($this->auditLogger);

        $this->loginHandler = new \RAP\LoginHandler($this->locatorStub, 'eduGAIN');
    }

    public function testTOUCheck(): void {
        $redirect = $this->loginHandler->onIdentityDataReceived('123', function($identity) {
            $identity->email = 'test@example.com';
        });

        $this->assertEquals('http://rap-ia2/tou-check', $redirect);
    }

    public function testExistingUserLogin(): void {

        $user = $this->getFakeUser();
        $this->assertEquals('test@example.com', $user->identities[0]->email);

        $this->userDaoStub->method('findUserByIdentity')->willReturn($user);

        $this->oAuth2RequestHandler->method('getRedirectResponseUrl')->willReturn('http://redirect-url');

        $this->sessionStub->method('getOAuth2RequestData')->willReturn(new \RAP\OAuth2RequestData());

        $this->userDaoStub->expects($this->once())
                ->method('updateIdentity')->with($this->anything());

        $redirect = $this->loginHandler->onIdentityDataReceived('123', function($identity) {
            $identity->email = 'test2@example.com';
        });

        // verify update
        $this->assertEquals('test2@example.com', $user->identities[0]->email);

        $this->assertEquals('http://redirect-url', $redirect);
    }

    public function testShowConfirmJoinNewIdentity(): void {

        $user = $this->getFakeUser();

        $this->sessionStub->method('getUser')->willReturn($user);
        $this->sessionStub->method('getAction')->willReturn('join');

        $redirect = $this->loginHandler->onIdentityDataReceived('456', function($identity) {
            $identity->email = 'test3@example.com';
        });

        $this->assertEquals('http://rap-ia2/confirm-join', $redirect);
    }

    public function testShowConfirmJoinExistingIdentity(): void {

        $user1 = $this->getFakeUser();

        $user2 = $this->getFakeUser();
        $user2->id = '2';
        $user2->identities[0]->id = '5';
        $user2->identities[0]->typedId = '456';

        $this->sessionStub->method('getUser')->willReturn($user1);
        $this->sessionStub->method('getAction')->willReturn('join');

        $this->userDaoStub->method('findUserByIdentity')->willReturn($user2);

        $redirect = $this->loginHandler->onIdentityDataReceived('456', function($identity) {
            $identity->email = 'test3@example.com';
        });

        $this->assertEquals('http://rap-ia2/confirm-join', $redirect);
    }

    public function testJoinExistingUser(): void {

        $user = $this->getFakeUser();
        $this->sessionStub->method('getUser')->willReturn($user);
        $this->sessionStub->method('getAction')->willReturn('join');

        $userToJoin = $this->getFakeUser();
        $userToJoin->id = '456';

        $redirect = $this->loginHandler->getAfterLoginRedirect($userToJoin);

        $this->assertEquals('http://rap-ia2/account', $redirect);
    }

    public function testJoinNewIdentity(): void {

        $user = $this->getFakeUser();
        $this->sessionStub->method('getUser')->willReturn($user);
        $this->sessionStub->method('getAction')->willReturn('join');

        $userToJoin = $this->getFakeUser();
        // new identity
        $userToJoin->id = null;

        $redirect = $this->loginHandler->getAfterLoginRedirect($userToJoin);

        $this->assertEquals('http://rap-ia2/account', $redirect);
    }

    public function testJoinAlreadyJoinedUser(): void { // go to account page without joining
        $user = $this->getFakeUser();

        $this->sessionStub->method('getAction')->willReturn('join');

        $this->userDaoStub->method('findUserByIdentity')->willReturn($user);

        $redirect = $this->loginHandler->onIdentityDataReceived('123', function($identity) {
            $identity->email = 'test@example.com';
        });

        $this->assertEquals('http://rap-ia2/account', $redirect);
    }

    private function getFakeUser(): \RAP\User {

        $user = new \RAP\User();
        $user->id = '1';
        $identity = new \RAP\Identity('eduGAIN');
        $identity->email = 'test@example.com';
        $identity->id = '4';
        $identity->typedId = '123';
        $user->addIdentity($identity);
        return $user;
    }

}
