<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

use \Firebase\JWT\JWT;

/**
 * See https://tools.ietf.org/html/rfc8693
 */
class TokenExchanger {

    private $locator;

    public function __construct(Locator $locator) {
        $this->locator = $locator;
    }

    public function exchangeToken(array $params, array $headers): array {

        $this->locator->getClientAuthChecker()->validateClientAuth($headers);

        if ($params['subject_token'] === null) {
            throw new BadRequestException("subject_token is required");
        }
        if ($params['subject_token_type'] === null) {
            throw new BadRequestException("subject_token_type is required");
        }
        if (strtolower($params['subject_token_type']) !== 'bearer') {
            throw new BadRequestException("subject_token_type " . $params['subject_token_type'] . " not supported");
        }

        $subjectToken = $this->locator->getTokenChecker()->getValidTokenObject($params['subject_token']);

        $claims = array(
            'sub' => $subjectToken->sub
        );

        if ($params['resource'] !== null) {
            $claims['resource'] = $params['resource'];
            $claims['jti'] = uniqid();
        }
        if ($params['audience'] !== null) {
            $claims['aud'] = $this->getAudienceClaim($params['audience']);
        }
        if ($params['scope'] !== null) {
            $claims['scope'] = $params['scope'];
        }
        if ($params['expires_in'] !== null) {
            $claims['exp'] = time() + intval($params['expires_in']);
        }
        
        $accessToken = $this->locator->getTokenBuilder()->generateToken($claims);
        
        $data = [];

        $data['access_token'] = $accessToken;
        $data['issued_token_type'] = "urn:ietf:params:oauth:token-type:jwt";
        $data['token_type'] = 'Bearer';
        $data['expires_in'] = $params['expires_in'] !== null ? $params['expires_in'] : 3600;

        return $data;
    }

    private function getAudienceClaim($audienceParam) {
        $audiences = explode(' ', $audienceParam);
        if (count($audiences) === 1) {
            // according to RFC 7519 audience can be a single value or an array
            return $audiences[0];
        }
        return $audiences;
    }

    /**
     * DEPRECATED (currently used by portals: to be removed)
     */
    public function exchangeTokenOld(string $token) {

        $key = $this->getExternalKeyForToken($token);
        $decoded = JWT::decode($token, $key->key, ['RS256']);

        $subject = $decoded->sub;
        $lifespan = ($decoded->exp - time());

        $data = [];

        $data['access_token'] = $this->locator->getTokenBuilder()->generateNewToken($subject, $lifespan / 3600, "gms");
        $data['issued_token_type'] = "urn:ietf:params:oauth:token-type:access_token";
        $data['token_type'] = 'Bearer';
        $data['expires_in'] = $lifespan;

        return $data;
    }

    private function getExternalKeyForToken(string $token): PublicJWK {

        $keys = $this->locator->getJWKSDAO()->getAllPublicJWK();

        $parts = explode('.', $token);
        $head = JWT::jsonDecode(JWT::urlsafeB64Decode($parts[0]));

        $kid = $head->kid;

        $key = $this->getKeyByKid($keys, $kid);
        if ($key === null) {
            $keys = $this->locator->getJWKSHandler()->loadAllJWKS();
        }
        $key = $this->getKeyByKid($keys, $kid);

        if ($key !== null) {
            return $key;
        }

        throw new \Exception("Invalid kid");
    }

    private function getKeyByKid(array $keys, string $kid): ?PublicJWK {
        foreach ($keys as $key) {
            if ($key->kid === $kid) {
                return $key;
            }
        }
        return null;
    }

}
