<?php

/* ----------------------------------------------------------------------------
 *               INAF - National Institute for Astrophysics
 *               IRA  - Radioastronomical Institute - Bologna
 *               OATS - Astronomical Observatory - Trieste
 * ----------------------------------------------------------------------------
 *
 * Copyright (C) 2016 Istituto Nazionale di Astrofisica
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License Version 3 as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

namespace RAP;

/**
 * This class wraps various objects that need to be stored into the session in
 * order to provide an object oriented transparent session management.
 */
class SessionData {

    const KEY = "SessionData";

    private $user;
    private $userToJoin;
    private $x509DataToRegister;
    private $oauth2RequestData;
    private $action;
    private $loginIdentityType;
    private $autojoin = false;

    public function setUser(?User $user): void {
        $this->user = $user;
        $this->save();
    }

    public function getUser(): ?User {
        return $this->user;
    }

    public function setUserToJoin(?User $userToJoin): void {
        $this->userToJoin = $userToJoin;
        $this->save();
    }

    public function getUserToJoin(): ?User {
        return $this->userToJoin;
    }

    public function setAutojoin(bool $autojoin): void {
        $this->autojoin = $autojoin;
        $this->save();
    }

    public function isAutojoin(): bool {
        return $this->autojoin;
    }

    public function setLoginIdentityType(string $loginIdentityType): void {
        $this->loginIdentityType = $loginIdentityType;
        $this->save();
    }

    public function getLoginIdentityType(): ?string {
        return $this->loginIdentityType;
    }

    /**
     * Update the user data model stored into the session after the primary
     * identity has changed, in order to avoid reading again the user data from
     * the database.
     * @param int $identityId
     */
    public function updatePrimaryIdentity($identityId): void {
        foreach ($this->user->identities as $identity) {
            $identity->primary = ($identity->id === $identityId);
        }
        $this->save();
    }

    public function setX509DataToRegister(?X509Data $x509DataToRegister): void {
        $this->x509DataToRegister = $x509DataToRegister;
        $this->save();
    }

    public function getX509DataToRegister(): ?X509Data {
        return $this->x509DataToRegister;
    }

    public function setOAuth2RequestData(?OAuth2RequestData $oauth2RequestData): void {
        $this->oauth2RequestData = $oauth2RequestData;
        $this->save();
    }

    public function getOAuth2RequestData(): ?OAuth2RequestData {
        return $this->oauth2RequestData;
    }

    public function setAction(?string $action): void {
        $this->action = $action;
        $this->save();
    }

    public function getAction(): ?string {
        return $this->action;
    }

    /**
     * Store the data into the $_SESSION PHP variable
     */
    public function save() {
        $_SESSION[SessionData::KEY] = $this;
    }

}
