<?php

use PHPUnit\Framework\TestCase;

final class OAuth2RequestHandlerTest extends TestCase {

    public function testBadRequestExceptionIfMissingClientId(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => null
        ];

        $locatorStub = $this->createMock(\RAP\Locator::class);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testInvalidClientRedirectURI(): void {

        $this->expectException(\RAP\BadRequestException::class);

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "invalid_redirect_uri",
            "state" => "state",
            "alg" => null,
            "nonce" => null,
            "scope" => "email%20profile"
        ];

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getBrowserBasedOAuth2ClientById')->willReturn(new \RAP\BrowserBasedOAuth2Client((object) [
                            "id" => "client_id",
                            "secret" => hash('sha256', "foo"),
                            "redirect" => "redirect_uri",
                            "scope" => "email profile",
                            "methods" => []
        ]));

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testExecuteOAuthStateFlow(): void {

        $params = [
            "client_id" => "client_id",
            "redirect_uri" => "redirect_uri",
            "state" => "state",
            "alg" => null,
            "nonce" => null,
            "scope" => "email%20profile"
        ];

        $client = new \RAP\BrowserBasedOAuth2Client((object) [
                    "id" => "client_id",
                    "secret" => hash('sha256', "foo"),
                    "redirect" => "redirect_uri",
                    "scope" => "email profile",
                    "methods" => []
        ]);
        $client->redirectUrl = "redirect_uri";

        $sessionStub = $this->createMock(\RAP\SessionData::class);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getBrowserBasedOAuth2ClientById')->willReturn($client);
        $locatorStub->method('getSession')->willReturn($sessionStub);

        $sessionStub->expects($this->once())
                ->method('setOAuth2RequestData')->with($this->anything());

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);
        $requestHandler->handleAuthorizeRequest($params);
    }

    public function testHandleCheckTokenRequest(): void {

        $tokenBuilderStub = $this->createMock(\RAP\TokenBuilder::class);
        $tokenBuilderStub->method('getIdToken')->willReturn('id-token');

        $tokenCheckerStub = $this->createMock(\RAP\TokenChecker::class);
        $tokenCheckerStub->method('validateToken')->willReturn((object) [
                    "sub" => "123",
                    "iat" => time(),
                    "exp" => time() + 3600,
                    "scope" => "openid email",
                    "aud" => "my-client"
        ]);

        $locatorStub = $this->createMock(\RAP\Locator::class);
        $locatorStub->method('getTokenChecker')->willReturn($tokenCheckerStub);
        $locatorStub->method('getTokenBuilder')->willReturn($tokenBuilderStub);

        $requestHandler = new \RAP\OAuth2RequestHandler($locatorStub);

        $result = $requestHandler->handleCheckTokenRequest(['Authorization' => 'Bearer: <token>']);

        $this->assertEquals(3600, $result['exp']);
        $this->assertEquals('123', $result['user_name']);
        $this->assertEquals('my-client', $result['client_id']);
        $this->assertEquals('id-token', $result['id_token']);
    }

}
