/*
 * This file is part of gms
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.DataSourceConfig;
import it.inaf.ia2.gms.HooksConfig;
import it.inaf.ia2.gms.model.GroupBreadcrumb;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.hook.GroupsHook;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import javax.sql.DataSource;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import org.mockito.internal.util.collections.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringRunner;

@RunWith(SpringRunner.class)
@ContextConfiguration(classes = {DataSourceConfig.class, HooksConfig.class})
public class GroupsDAOTest {

    @Autowired
    private DataSource dataSource;

    @Autowired
    @SpyBean
    private GroupsHook groupsHook;

    private GroupsDAO dao;

    @Before
    public void setUp() {
        dao = new GroupsDAO(dataSource);
    }

    @Test
    public void testAll() {

        // Create groups
        GroupEntity root = new GroupEntity();
        root.setId("ROOT");
        root.setName("ROOT");
        root.setPath("");
        dao.createGroup(root);

        dao.groupsHook = groupsHook;

        GroupEntity lbt = new GroupEntity();
        lbt.setId(getNewGroupId());
        lbt.setName("LBT");
        lbt.setPath(lbt.getId());
        dao.createGroup(lbt);

        verify(groupsHook, times(1)).beforeCreate(eq(lbt));

        GroupEntity tng = new GroupEntity();
        tng.setId(getNewGroupId());
        tng.setName("TNG");
        tng.setPath(tng.getId());
        dao.createGroup(tng);

        verify(groupsHook, times(1)).beforeCreate(eq(tng));

        GroupEntity lbtInaf = new GroupEntity();
        lbtInaf.setId(getNewGroupId());
        lbtInaf.setName("INAF");
        lbtInaf.setPath(lbt.getId() + "." + lbtInaf.getId());
        dao.createGroup(lbtInaf);

        GroupEntity lbtInafP1 = new GroupEntity();
        lbtInafP1.setId(getNewGroupId());
        lbtInafP1.setName("P1");
        lbtInafP1.setPath(lbt.getId() + "." + lbtInaf.getId() + "." + lbtInafP1.getId());
        dao.createGroup(lbtInafP1);

        // Check count
        assertEquals(5, dao.count());

        // Find by id
        Optional<GroupEntity> group = dao.findGroupById(lbtInaf.getId());
        assertTrue(group.isPresent());
        assertEquals(lbtInaf, group.get());

        group = dao.findGroupByPath(lbtInaf.getPath());
        assertTrue(group.isPresent());
        assertEquals(lbtInaf, group.get());

        // Sub list
        List<GroupEntity> groups = dao.getDirectSubGroups(root.getPath());
        assertEquals(2, groups.size());
        assertEquals("LBT", groups.get(0).getName());
        assertEquals("TNG", groups.get(1).getName());

        groups = dao.getDirectSubGroups(lbt.getId());
        assertEquals(1, groups.size());
        assertEquals("INAF", groups.get(0).getName());

        // All children
        groups = dao.getAllChildren(root.getPath());
        assertEquals(4, groups.size());
        assertEquals("P1", groups.get(0).getName()); // order by path DESC
        assertEquals("INAF", groups.get(1).getName());

        // Group by parent and name
        Optional<GroupEntity> optGroup = dao.findGroupByParentAndName(root.getPath(), lbt.getName());
        assertTrue(optGroup.isPresent());
        assertEquals(lbt.getId(), optGroup.get().getId());

        optGroup = dao.findGroupByParentAndName(lbt.getPath(), lbtInaf.getName());
        assertTrue(optGroup.isPresent());
        assertEquals(lbtInaf.getId(), optGroup.get().getId());

        // Complete names
        Set<String> groupIds = new HashSet<>();
        groupIds.add(groups.get(0).getId());
        groupIds.add(lbt.getId());
        Map<String, String> completeGroupNames = dao.getGroupCompleteNamesFromId(groupIds);
        assertEquals(2, completeGroupNames.size());
        assertEquals("LBT", completeGroupNames.get(lbt.getId()));
        assertEquals("LBT.INAF.P1", completeGroupNames.get(groups.get(0).getId()));

        // Children map
        Map<String, Boolean> childrenMap = dao.getHasChildrenMap(Sets.newSet(root.getId()));
        assertEquals(1, childrenMap.size());
        assertTrue(childrenMap.get(root.getId()));

        childrenMap = dao.getHasChildrenMap(Sets.newSet(lbt.getId(), tng.getId()));
        assertEquals(2, childrenMap.size());
        assertTrue(childrenMap.get(lbt.getId()));
        assertFalse(childrenMap.get(tng.getId()));

        // Update
        String newName = "renamed";
        tng.setName(newName);
        dao.updateGroup(tng);
        tng = dao.findGroupById(tng.getId()).get();
        assertEquals(newName, tng.getName());

        verify(groupsHook, times(1)).beforeUpdate(eq(tng));

        // Breadcrumbs
        List<GroupBreadcrumb> breadcrumbs = dao.getBreadcrumbs(lbt.getId() + "." + lbtInaf.getId());
        assertEquals(3, breadcrumbs.size());
        assertEquals(root.getName(), breadcrumbs.get(0).getGroupName());
        assertEquals(lbt.getName(), breadcrumbs.get(1).getGroupName());
        assertEquals(lbtInaf.getName(), breadcrumbs.get(2).getGroupName());

        // Search
        assertEquals(1, dao.searchGroups("renam").size());
        assertTrue(dao.searchGroups("invalid-name").isEmpty());

        // Find by names
        List<GroupEntity> groupsByName = dao.findGroupsByNames(Arrays.asList("LBT", "INAF"));
        assertEquals(2, groupsByName.size());
        assertEquals("LBT", groupsByName.get(0).getName());
        assertEquals("INAF", groupsByName.get(1).getName());

        // Find by ids
        assertEquals(2, dao.findGroupsByIds(new HashSet<>(Arrays.asList(lbt.getId(), tng.getId()))).size());

        // Get direct children
        List<GroupEntity> lbtChildren = dao.getDirectSubGroups(lbt.getPath());
        assertEquals(1, lbtChildren.size());
        assertEquals("INAF", lbtChildren.get(0).getName());

        // Get direct children with filter
        assertEquals(1, dao.getDirectSubGroups(lbt.getPath(), "INA").size());
        assertTrue(dao.getDirectSubGroups(lbt.getPath(), "invalid-name").isEmpty());

        // Delete
        dao.deleteGroup(lbtInaf);
        groups = dao.getDirectSubGroups(lbt.getId());
        assertTrue(groups.isEmpty());

        verify(groupsHook, times(1)).beforeDelete(eq(lbtInaf));
    }

    private String getNewGroupId() {
        return UUID.randomUUID().toString().replaceAll("-", "");
    }

    @Test
    public void testFields() {

        GroupEntity group = new GroupEntity();
        group.setId("group_id");
        group.setName("group_name");
        group.setPath("group_path");
        group.setLeaf(true);
        group.setLocked(true);
        group.setCreatedBy("creator_id");

        dao.createGroup(group);

        GroupEntity savedGroup = dao.findGroupById("group_id").get();

        assertEquals("group_id", savedGroup.getId());
        assertEquals("group_name", savedGroup.getName());
        assertEquals("group_path", savedGroup.getPath());
        assertTrue(savedGroup.isLeaf());
        assertTrue(savedGroup.isLocked());
        assertEquals("creator_id", savedGroup.getCreatedBy());

        group.setName("new_name");
        group.setLeaf(false);
        group.setLocked(false);

        dao.updateGroup(group);

        savedGroup = dao.findGroupById("group_id").get();

        assertEquals("new_name", savedGroup.getName());
        assertFalse(savedGroup.isLeaf());
        assertFalse(savedGroup.isLocked());
    }

    @Test
    public void testGetInexistentGroupById() {
        assertTrue(dao.findGroupById("not-found").isEmpty());
    }

    @Test
    public void testGroupCompleteNamesEmptyInput() {
        assertTrue(dao.getGroupCompleteNamesFromId(new HashSet<>()).isEmpty());
    }

    @Test
    public void testGetHasChildrenMapEmptyInput() {
        assertTrue(dao.getHasChildrenMap(new HashSet<>()).isEmpty());
    }

    @Test
    public void testFindGroupsByIdsEmtpyInput() {
        assertTrue(dao.findGroupsByIds(new HashSet<>()).isEmpty());
    }

    @Test
    public void findInexistentGroupByParentAndName() {
        assertTrue(dao.findGroupByParentAndName("", "not-exists").isEmpty());
    }

    @Test
    public void findInexistentGroupByPath() {
        assertTrue(dao.findGroupByPath("not.exists").isEmpty());
    }
}
