/*
 * This file is part of gms
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import java.sql.PreparedStatement;
import java.sql.Types;
import java.util.Collections;
import java.util.Set;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

@Component
public class JoinDAO {

    private static final Logger LOG = LoggerFactory.getLogger(JoinDAO.class);

    private final JdbcTemplate jdbcTemplate;

    @Autowired
    public JoinDAO(DataSource dataSource) {
        jdbcTemplate = new JdbcTemplate(dataSource);
    }

    @Transactional
    public void join(Set<MembershipEntity> membershipsToAdd, Set<PermissionEntity> permissionsToAdd, String userToDelete) {

        if (!membershipsToAdd.isEmpty()) {
            addMemberships(membershipsToAdd);
        }
        if (!permissionsToAdd.isEmpty()) {
            addPermissions(permissionsToAdd);
        }
        deleteUserMemberships(userToDelete);
        deleteUserPermissions(userToDelete);
    }

    private void addMemberships(Set<MembershipEntity> membershipsToAdd) {

        String sql = "INSERT INTO gms_membership (group_id, user_id) VALUES "
                + String.join(", ", Collections.nCopies(membershipsToAdd.size(), "(?, ?)"));

        LOG.trace("Executing {}", sql);

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            int i = 0;
            for (MembershipEntity membership : membershipsToAdd) {
                ps.setString(++i, membership.getGroupId());
                ps.setString(++i, membership.getUserId());
            }
            return ps;
        });
    }

    private void addPermissions(Set<PermissionEntity> permissionsToAdd) {

        String sql = "INSERT INTO gms_permission (group_id, user_id, permission, group_path, set_by) VALUES "
                + String.join(", ", Collections.nCopies(permissionsToAdd.size(), "(?, ?, ?, ?, ?)")) + "\n"
                + "ON CONFLICT (group_id, user_id) DO UPDATE\n"
                + "SET permission = EXCLUDED.permission, set_by = EXCLUDED.set_by, update_time = NOW()";

        LOG.trace("Executing {}", sql);

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            int i = 0;
            for (PermissionEntity permission : permissionsToAdd) {
                ps.setString(++i, permission.getGroupId());
                ps.setString(++i, permission.getUserId());
                ps.setObject(++i, permission.getPermission().toString(), Types.OTHER);
                ps.setObject(++i, permission.getGroupPath(), Types.OTHER);
                ps.setString(++i, permission.getSetBy());
            }
            return ps;
        });
    }

    private void deleteUserMemberships(String userId) {

        String sql = "DELETE FROM gms_membership WHERE user_id = ?";

        LOG.trace("Executing {}", sql);

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, userId);
            return ps;
        });
    }

    private void deleteUserPermissions(String userId) {

        String sql = "DELETE FROM gms_permission WHERE user_id = ?";

        LOG.trace("Executing {}", sql);

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, userId);
            return ps;
        });
    }
}
