/*
 * This file is part of gms
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.request.AddPermissionRequest;
import it.inaf.ia2.gms.model.request.MemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUserPermission;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.model.request.UpdatePermissionRequest;
import it.inaf.ia2.gms.model.response.UserPermission;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupNameService;
import it.inaf.ia2.gms.service.SearchService;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class PermissionsController {

    @Autowired
    private GroupsManager groupsManager;

    @Autowired
    private PermissionsManager permissionsManager;

    @Autowired
    protected GroupNameService groupNameService;

    @Autowired
    private SearchService searchService;

    @GetMapping(value = "/ui/permissions", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUserPermission>> getPermissionsTab(TabRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        PaginatedData<RapUserPermission> permissionsPanel = getPermissionsPanel(group, request);

        return ResponseEntity.ok(permissionsPanel);
    }

    @GetMapping(value = "/ui/permission", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Map<String, Permission>> getUserPermission(@RequestParam("groupId") String groupId, @RequestParam("userId") String userId) {

        GroupEntity group = groupsManager.getGroupById(groupId);

        Permission permission = permissionsManager.getDirectUserPermission(group, userId);
        if (permission == null) {
            return ResponseEntity.noContent().build();
        } else {
            Map<String, Permission> response = new HashMap<>();
            response.put("permission", permission);
            return ResponseEntity.ok(response);
        }
    }

    @PostMapping(value = "/ui/permission", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUserPermission>> addPermission(@Valid @RequestBody AddPermissionRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        if (request.isOverride()) {
            permissionsManager.updatePermission(group, request.getUserId(), request.getPermission());
        } else {
            permissionsManager.addPermission(group, request.getUserId(), request.getPermission());
        }

        return new ResponseEntity<>(getPermissionsPanel(group, request), HttpStatus.CREATED);
    }

    @PutMapping(value = "/ui/permission", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Map<String, String>> updatePermission(@Valid @RequestBody UpdatePermissionRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        PermissionEntity updatedEntity = permissionsManager.updatePermission(group, request.getUserId(), request.getPermission());

        Map<String, String> response = new HashMap<>();
        response.put("permission", updatedEntity.getPermission().toString());

        return ResponseEntity.ok(response);
    }

    @DeleteMapping(value = "/ui/permission", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUserPermission>> deletePermission(@Valid MemberRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        permissionsManager.removePermission(group, request.getUserId());

        return ResponseEntity.ok(getPermissionsPanel(group, request));
    }

    private PaginatedData<RapUserPermission> getPermissionsPanel(GroupEntity group, PaginatedModelRequest request) {
        List<RapUserPermission> permissions = permissionsManager.getAllPermissions(group);
        Collections.sort(permissions, (p1, p2) -> {
            return p1.getUser().getDisplayName().compareTo(p2.getUser().getDisplayName());
        });
        return new PaginatedData<>(permissions, request.getPaginatorPage(), request.getPaginatorPageSize());
    }

    @GetMapping(value = "/permission", produces = MediaType.TEXT_PLAIN_VALUE)
    public void getUserPermission(@RequestParam("group") Optional<String> groupNames, @RequestParam("user_id") Optional<String> userId, HttpServletRequest request, HttpServletResponse response) throws IOException {

        GroupEntity groupEntity = groupNameService.getGroupFromNames(groupNames);
        if (userId.isPresent()) {
            if (userId.get().equals(request.getUserPrincipal().getName())) {
                // asking my permissions
                try ( PrintWriter pw = new PrintWriter(response.getOutputStream())) {
                    List<PermissionEntity> userPermissions = permissionsManager.getCurrentUserPermissions();

                    Map<String, List<String>> namesMap = groupNameService.getNamesFromIds(userPermissions.stream()
                            .map(pe -> pe.getGroupId()).collect(Collectors.toSet()));

                    for (PermissionEntity pe : userPermissions) {
                        pw.println(groupNameService.getCompleteName(namesMap.get(pe.getGroupId())) + " " + pe.getPermission());
                    }
                }
            } else {
                try ( PrintWriter pw = new PrintWriter(response.getOutputStream())) {
                    for (UserPermission userPermission : searchService.getUserPermission(groupEntity, userId.get(), permissionsManager.getCurrentUserPermissions(groupEntity))) {
                        String group = groupNameService.getCompleteName(userPermission.getGroupCompleteName());
                        pw.println(group + " " + userPermission.getPermission());
                    }
                }
            }
        } else {
            try ( PrintWriter pw = new PrintWriter(response.getOutputStream())) {
                for (it.inaf.ia2.gms.model.RapUserPermission up : permissionsManager.getAllPermissions(groupEntity)) {
                    pw.println(up.getUser().getId() + " " + up.getPermission());
                }
            }
        }
    }

    @PostMapping(value = "/permission", produces = MediaType.TEXT_PLAIN_VALUE, consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE)
    public void addPermission(@RequestParam("group") Optional<String> groupNames, @RequestParam("user_id") String targetUserId, @RequestParam("permission") Permission permission) throws IOException {
        GroupEntity groupEntity = groupNameService.getGroupFromNames(groupNames);
        permissionsManager.addPermission(groupEntity, targetUserId, permission);
    }

    @PutMapping(value = "/permission", produces = MediaType.TEXT_PLAIN_VALUE, consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE)
    public void setPermission(@RequestParam("group") Optional<String> groupNames, @RequestParam("user_id") String targetUserId, @RequestParam("permission") Permission permission) throws IOException {
        GroupEntity groupEntity = groupNameService.getGroupFromNames(groupNames);
        permissionsManager.createOrUpdatePermission(groupEntity, targetUserId, permission);
    }

    @DeleteMapping(value = "/permission", produces = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<?> removePermission(@RequestParam("group") Optional<String> groupNames, @RequestParam("user_id") String userId) {

        GroupEntity groupEntity = groupNameService.getGroupFromNames(groupNames);
        permissionsManager.removePermission(groupEntity, userId);

        return ResponseEntity.noContent().build();
    }
}
