/*
 * This file is part of gms-client
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.client;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.CompletableFuture;
import org.mockito.ArgumentMatcher;
import org.mockito.ArgumentMatchers;
import static org.mockito.ArgumentMatchers.any;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class BaseGmsClientTest {

    private static final String BASE_URL = "http://base-url";

    protected HttpClient httpClient;
    protected GmsClient gmsClient;

    public void init() {
        httpClient = mock(HttpClient.class);
        gmsClient = getMockedGmsClient(httpClient);
    }

    protected static String getResourceFileContent(String fileName) {
        try ( InputStream in = BaseGmsClientTest.class.getClassLoader().getResourceAsStream(fileName)) {
            return new String(in.readAllBytes(), StandardCharsets.UTF_8);
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    protected static GmsClient getMockedGmsClient(HttpClient mockedHttpClient) {

        HttpClient.Builder builder = mock(HttpClient.Builder.class);
        when(builder.followRedirects(any())).thenReturn(builder);
        when(builder.version(any())).thenReturn(builder);
        when(builder.build()).thenReturn(mockedHttpClient);

        try ( MockedStatic<HttpClient> staticMock = Mockito.mockStatic(HttpClient.class)) {
            staticMock.when(HttpClient::newBuilder).thenReturn(builder);
            return new GmsClient(BASE_URL).setAccessToken("foo");
        }
    }

    protected static CompletableFuture<HttpResponse<InputStream>> getMockedStreamResponseFuture(int statusCode, String body) {
        return CompletableFuture.completedFuture(getMockedStreamResponse(200, body));
    }

    protected static CompletableFuture<HttpResponse<String>> getMockedStringResponseFuture(int statusCode, String body) {
        return CompletableFuture.completedFuture(getMockedStringResponse(200, body));
    }

    protected static HttpResponse<InputStream> getMockedStreamResponse(int statusCode, String body) {
        HttpResponse response = getMockedResponse(statusCode);
        InputStream in = new ByteArrayInputStream(body.getBytes(StandardCharsets.UTF_8));
        when(response.body()).thenReturn(in);
        return response;
    }

    protected static HttpResponse<String> getMockedStringResponse(int statusCode, String body) {
        HttpResponse response = getMockedResponse(statusCode);
        when(response.body()).thenReturn(body);
        return response;
    }

    protected static HttpResponse getMockedResponse(int statusCode) {
        HttpResponse response = mock(HttpResponse.class);
        when(response.statusCode()).thenReturn(statusCode);
        return response;
    }

    protected static HttpRequest endpointEq(String expectedMethod, String expectedEndpoint) {
        return ArgumentMatchers.argThat(endpointEqArgumentMatcher(expectedMethod, expectedEndpoint));
    }

    protected static ArgumentMatcher<HttpRequest> endpointEqArgumentMatcher(String expectedMethod, String expectedEndpoint) {

        return new ArgumentMatcher<HttpRequest>() {

            private final String expectedUri = BASE_URL + "/" + expectedEndpoint;

            @Override
            public boolean matches(HttpRequest request) {
                return expectedMethod.equals(request.method()) && expectedUri.equals(request.uri().toString());
            }

            @Override
            public String toString() {
                return expectedMethod + " " + expectedUri;
            }
        };
    }
}
