package it.inaf.ia2.gms.manager;

import it.inaf.ia2.gms.GmsTestUtils;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.InvitedRegistrationDAO;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.InvitedRegistration;
import it.inaf.ia2.gms.service.PermissionsService;
import it.inaf.ia2.rap.client.RapClient;
import it.inaf.ia2.rap.data.Identity;
import it.inaf.ia2.rap.data.IdentityType;
import it.inaf.ia2.rap.data.RapUser;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class InvitedRegistrationManagerTest {

    private static final String USER_ID = "USER_ID";
    private static final String EMAIL = "user@inaf.it";

    @Mock
    private GroupsDAO groupsDAO;
    @Mock
    private MembershipsDAO membershipsDAO;
    @Mock
    private PermissionsService permissionsService;
    @Mock
    private PermissionsManager permissionsManager;
    @Mock
    private InvitedRegistrationDAO invitedRegistrationDAO;
    @Mock
    private RapClient rapClient;
    @Mock
    private HttpServletRequest servletRequest;
    @Mock
    private LoggingDAO loggingDAO;
    @Mock
    private HttpSession httpSession;

    @InjectMocks
    private InvitedRegistrationManager invitedRegistrationManager;

    @Before
    public void setUp() {
        UserAwareComponentTestUtil.setUser(invitedRegistrationManager, USER_ID);
    }

    @Test
    public void testAddInvitedRegistration() {

        Map<GroupEntity, Permission> groupsPermissions = new HashMap<>();
        GroupEntity groupEntity = new GroupEntity();
        groupEntity.setId("b");
        groupEntity.setPath("a.b");
        groupEntity.setName("B");
        groupsPermissions.put(groupEntity, Permission.MANAGE_MEMBERS);

        when(permissionsManager.getCurrentUserPermission(any())).thenReturn(Permission.ADMIN);

        invitedRegistrationManager.addInvitedRegistration("token_hash", EMAIL, groupsPermissions);

        verify(invitedRegistrationDAO, times(1)).addInvitedRegistration(any());
    }

    @Test
    public void testCompleteInvitedRegistrationIfNecessaryGetByEmail() {

        Map<String, Permission> map1 = new HashMap<>();
        map1.put("g1", Permission.MANAGE_MEMBERS);

        InvitedRegistration regFromToken = new InvitedRegistration()
                .setId("reg1")
                .setEmail(EMAIL)
                .setGroupsPermissions(map1);

        when(httpSession.getAttribute(eq("invited-registration"))).thenReturn(regFromToken);

        GmsTestUtils.mockPrincipal(servletRequest, USER_ID);

        RapUser user = new RapUser();
        user.setId(USER_ID);
        Identity identity = new Identity();
        identity.setType(IdentityType.EDU_GAIN);
        identity.setEmail(EMAIL);
        user.getIdentities().addAll(Collections.singletonList(identity));

        when(rapClient.getUser(eq(USER_ID))).thenReturn(user);

        Map<String, Permission> map2 = new HashMap<>();
        map2.put("g2", Permission.MANAGE_MEMBERS);

        InvitedRegistration reg2 = new InvitedRegistration()
                .setId("reg2")
                .setEmail(EMAIL)
                .setGroupsPermissions(map2);

        List<InvitedRegistration> registrations = new ArrayList<>();
        registrations.add(regFromToken);
        registrations.add(reg2);
        when(invitedRegistrationDAO.getInvitedRegistrationsFromEmailAddresses(any())).thenReturn(registrations);

        GroupEntity g1 = new GroupEntity();
        g1.setId("g1");
        GroupEntity g2 = new GroupEntity();
        g2.setId("g2");
        when(groupsDAO.findGroupById(eq("g1"))).thenReturn(Optional.of(g1));
        when(groupsDAO.findGroupById(eq("g2"))).thenReturn(Optional.of(g2));

        invitedRegistrationManager.completeInvitedRegistrationIfNecessary();

        verify(membershipsDAO, times(2)).addMember(any());
        verify(permissionsService, times(2)).addPermission(any(), any(), any());
    }

    @Test
    public void testCompleteInvitedRegistrationIfNecessaryGetByToken() {

        Map<String, Permission> map1 = new HashMap<>();
        map1.put("g1", Permission.MANAGE_MEMBERS);

        InvitedRegistration regFromToken = new InvitedRegistration()
                .setId("reg1")
                .setEmail(EMAIL)
                .setGroupsPermissions(map1);

        when(httpSession.getAttribute(eq("invited-registration"))).thenReturn(regFromToken);

        GmsTestUtils.mockPrincipal(servletRequest, USER_ID);

        RapUser user = new RapUser();
        user.setId(USER_ID);
        Identity identity = new Identity();
        identity.setType(IdentityType.EDU_GAIN);
        identity.setEmail("different-email");
        user.setIdentities(Collections.singletonList(identity));

        when(rapClient.getUser(eq(USER_ID))).thenReturn(user);

        GroupEntity g1 = new GroupEntity();
        g1.setId("g1");
        when(groupsDAO.findGroupById(eq("g1"))).thenReturn(Optional.of(g1));

        invitedRegistrationManager.completeInvitedRegistrationIfNecessary();

        verify(membershipsDAO, times(1)).addMember(any());
        verify(permissionsService, times(1)).addPermission(any(), any(), any());
    }
}
