package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.manager.MembershipManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.request.AddMemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.request.RemoveMemberRequest;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.rap.data.RapUser;
import java.util.Collections;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class MembersController {

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private MembershipManager membershipManager;

    @Autowired
    private PermissionsManager permissionsManager;

    @GetMapping(value = "/members", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> getMembersTab(TabRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        PaginatedData<RapUser> membersPanel = getMembersPanel(group, request);

        return ResponseEntity.ok(membersPanel);
    }

    @PostMapping(value = "/member", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> addMember(@Valid @RequestBody AddMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        membershipManager.addMember(group, request.getUserId());
        permissionsManager.addPermission(group, request.getUserId(), request.getPermission());

        return new ResponseEntity<>(getMembersPanel(group, request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/member", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> removeMember(@Valid RemoveMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        membershipManager.removeMember(group, request.getUserId());

        if (request.isRemoveAlsoPermission()) {
            permissionsManager.removePermission(group, request.getUserId());
        }

        return ResponseEntity.ok(getMembersPanel(group, request));
    }

    private PaginatedData<RapUser> getMembersPanel(GroupEntity group, PaginatedModelRequest request) {
        List<RapUser> members = membershipManager.getMembers(group);
        Collections.sort(members, (m1, m2) -> {
            return m1.getDisplayName().compareTo(m2.getDisplayName());
        });
        return new PaginatedData<>(members, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
