package it.inaf.ia2.gms.cli;

import it.inaf.ia2.client.ClientException;
import it.inaf.ia2.gms.client.GmsClient;
import it.inaf.ia2.gms.client.model.Permission;
import it.inaf.ia2.rap.client.RapClient;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.util.List;
import java.util.Properties;

public class CLI {

    public static void main(String[] args) throws Exception {
        new CLI(args).run();
    }

    private final String args[];
    private int argIndex;

    private String gmsBaseUrl;
    private String rapBaseUrl;
    private String clientId;
    private String clientSecret;
    private String token;

    private GmsClient client;

    private CLI(String... args) {
        this.args = args;
    }

    private void run() throws Exception {
        if (args.length < 2) {
            displayUsage();
        }

        boolean commandParsed = false;
        while (argIndex < args.length && !commandParsed) {
            switch (args[argIndex]) {
                case "--config-file":
                    loadConfigFromFile(new File(getNextArg()));
                    break;
                case "--token-file":
                    loadTokenFromFile(new File(getNextArg()));
                    break;
                case "--gms-url":
                    gmsBaseUrl = getNextArg();
                    break;
                case "--rap-url":
                    rapBaseUrl = getNextArg();
                    break;
                case "--client-id":
                    clientId = getNextArg();
                    break;
                case "--client-secret":
                    clientSecret = getNextArg();
                    break;
                default:
                    verifyConfigLoaded();
                    createClient();
                    try {
                        parseCommand();
                    } catch (ClientException ex) {
                        System.err.println(ex.getMessage());
                    }
                    commandParsed = true;
                    break;
            }
            argIndex++;
        }
    }

    private String getNextArg() {
        if (argIndex + 1 == args.length) {
            System.err.println("Missing value for option " + args[argIndex]);
            System.exit(1);
        }
        return args[++argIndex];
    }

    private void verifyConfigLoaded() {

        if (gmsBaseUrl == null) {
            // Attempt reading gms.properties in current directory
            loadConfigFromFile(new File("gms.properties"));
        }

        if (clientId == null && token == null) {
            // Attempt loading token.txt in current directory
            loadTokenFromFile(new File("token.txt"));
        }

        if (token != null && (clientSecret == null || rapBaseUrl == null)) {
            System.err.println("Client secret and RAP base URL not configured");
            System.exit(1);
        }
    }

    private void createClient() {

        client = new GmsClient(gmsBaseUrl);

        if (token != null) {
            client.setAccessToken(token);
        } else {            
            RapClient rapClient = new RapClient(rapBaseUrl)
                    .setClientId(clientId)
                    .setClientSecret(clientSecret);            
            client.setAccessToken(rapClient.getAccessTokenFromClientCredentials());
        }
    }

    private void loadConfigFromFile(File config) {

        if (!config.exists()) {
            System.err.println("Config file " + config.getAbsolutePath() + " doesn't exist");
            System.exit(1);
        }

        Properties properties = new Properties();
        try ( InputStream in = new FileInputStream(config)) {
            properties.load(in);
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }

        gmsBaseUrl = properties.getProperty("gms_url");
        if (gmsBaseUrl == null) {
            System.err.println("Missing gms_url in gms.properties");
            System.exit(1);
        }
        rapBaseUrl = properties.getProperty("rap_url");
        clientId = properties.getProperty("client_id");
        clientSecret = properties.getProperty("client_secret");
    }

    private void loadTokenFromFile(File tokenFile) {

        if (!tokenFile.exists()) {
            System.err.println("Token file " + tokenFile.getAbsolutePath() + " doesn't exist");
            System.exit(1);
        }

        try ( InputStream in = new FileInputStream(tokenFile)) {
            java.util.Scanner s = new java.util.Scanner(in).useDelimiter("\\A");
            token = s.next().trim();
        } catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    private void parseCommand() {

        switch (args[argIndex]) {
            case "create-group":
                boolean leaf = false;
                if (argIndex + 2 < args.length) {
                    leaf = Boolean.parseBoolean(args[argIndex + 2]);
                }
                client.createGroup(args[argIndex + 1], leaf);
                System.out.println("Group created");
                break;
            case "delete-group":
                client.deleteGroup(args[argIndex + 1]);
                System.out.println("Group deleted");
                break;
            case "add-member":
                if (argIndex + 2 >= args.length) {
                    displayUsage();
                }
                client.addMember(args[argIndex + 1], args[argIndex + 2]);
                System.out.println("Member added");
                break;
            case "remove-member":
                if (argIndex + 2 >= args.length) {
                    displayUsage();
                }
                client.removeMember(args[argIndex + 1], args[argIndex + 2]);
                System.out.println("Member removed");
                break;
            case "set-permission":
                if (argIndex + 3 >= args.length) {
                    displayUsage();
                }
                client.setPermission(args[argIndex + 1], args[argIndex + 2], Permission.valueOf(args[argIndex + 3]));
                System.out.println("Permission changed");
                break;
            case "add-permission":
                if (argIndex + 3 >= args.length) {
                    displayUsage();
                }
                client.addPermission(args[argIndex + 1], args[argIndex + 2], Permission.valueOf(args[argIndex + 3]));
                System.out.println("Permission added");
                break;
            case "delete-permission":
                if (argIndex + 2 >= args.length) {
                    displayUsage();
                }
                client.removePermission(args[argIndex + 1], args[argIndex + 2]);
                System.out.println("Permission removed");
                break;
            case "get-member-email-addresses":
                Permission permission = null;
                if (argIndex + 2 < args.length) {
                    permission = Permission.valueOf(args[argIndex + 2]);
                }
                List<String> addresses = client.getMemberEmailAddresses(args[argIndex + 1], permission);
                for (String address : addresses) {
                    System.out.println(address);
                }
                break;
            default:
                displayUsage();
                break;
        }
    }

    private void displayUsage() {
        System.out.println("gms-client\n"
                + "    [--config-file <file>]\n"
                + "    [--token-file <file>]\n"
                + "    [--gms-url <url>]\n"
                + "    [--rap-url <url>]\n"
                + "    [--client-id <id>]\n"
                + "    [--client-secret <secret>]\n"
                + "    create-group <name1.name2.name3> [<leaf>]\n"
                + "    delete-group <name1.name2.name3>\n"
                + "    add-member <name1.name2.name3> <user_id>\n"
                + "    remove-member <name1.name2.name3> <user_id>\n"
                + "    set-permission <name1.name2.name3> <user_id> <permission>\n"
                + "    add-permission <name1.name2.name3> <user_id> <permission>\n"
                + "    delete-permission <name1.name2.name3> <user_id>\n"
                + "    get-member-email-addresses <name1.name2.name3> [<permission>]");
        System.exit(0);
    }
}
