package it.inaf.ia2.gms.persistence;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.sql.PreparedStatement;
import javax.servlet.http.HttpServletRequest;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

@Component
public class LoggingDAO {

    private static final Logger LOG = LoggerFactory.getLogger(LoggingDAO.class);

    private final JdbcTemplate jdbcTemplate;

    @Autowired(required = false)
    private HttpServletRequest request;

    @Autowired
    public LoggingDAO(DataSource dataSource) {
        jdbcTemplate = new JdbcTemplate(dataSource);
    }

    public void logException(Exception ex) {

        try {
            String sql = "INSERT INTO error_log (exception_class, stack_trace) VALUES (?, ?)";

            jdbcTemplate.update(conn -> {
                PreparedStatement ps = conn.prepareStatement(sql);
                ps.setString(1, ex.getClass().getCanonicalName());
                ps.setString(2, getStackTraceString(ex));
                return ps;
            });

        } catch (Throwable throwable) {
            LOG.error("Exception while trying to save log exception", throwable);
            LOG.error("Original exception", ex);
        }
    }

    private String getStackTraceString(Exception ex) {
        StringWriter sw = new StringWriter();
        try (PrintWriter pw = new PrintWriter(sw)) {
            ex.printStackTrace(pw);
        }
        return sw.toString();
    }

    public void logAction(String action) {
        logAction(action, request);
    }

    public void logAction(String action, HttpServletRequest request) {
        try {
            String sql = "INSERT INTO audit_log (\"user\", action, ip_address) VALUES (?, ?, ?)";

            jdbcTemplate.update(conn -> {
                PreparedStatement ps = conn.prepareStatement(sql);
                int i = 0;
                ps.setString(++i, getUser(request));
                ps.setString(++i, action);
                ps.setString(++i, getIPAddress(request));
                return ps;
            });
        } catch (Throwable t) {
            LOG.error("Exception while trying to save audit log", t);
        }
    }

    private String getIPAddress(HttpServletRequest request) {
        String ipAddress = request.getHeader("X-FORWARDED-FOR");
        if (ipAddress == null) {
            return request.getRemoteAddr();
        } else {
            return ipAddress.contains(",") ? ipAddress.split(",")[0] : ipAddress;
        }
    }

    private String getUser(HttpServletRequest request) {
        if (request.getUserPrincipal() != null) {
            return request.getUserPrincipal().getName();
        }
        return null;
    }
}
