package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.gms.GmsTestUtils;
import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.request.AddGroupRequest;
import it.inaf.ia2.gms.model.response.GroupsTabResponse;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupNameService;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.GroupsTreeBuilder;
import java.util.ArrayList;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.delete;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class GroupsControllerTest {

    @Mock
    private GroupsTabResponseBuilder groupsTabResponseBuilder;

    @Mock
    private GroupsManager groupsManager;

    @Mock
    private GroupsService groupsService;

    @Mock
    private HttpServletRequest servletRequest;

    @Mock
    private PermissionsManager permissionsManager;

    @Mock
    private GroupsTreeBuilder groupsTreeBuilder;

    @Mock
    private GroupsDAO groupsDAO;

    @Mock
    private LoggingDAO loggingDAO;

    @InjectMocks
    private GroupsController controller;

    private final ObjectMapper mapper = new ObjectMapper();

    private MockMvc mockMvc;

    @Before
    public void init() {
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();
        controller.groupNameService = new GroupNameService(groupsDAO);
        GmsTestUtils.mockPrincipal(servletRequest);
    }

    @Test
    public void testGetGroups() throws Exception {

        GroupsTabResponse response = new GroupsTabResponse();
        response.setBreadcrumbs(new ArrayList<>());
        response.setGroupsPanel(new PaginatedData<>(new ArrayList<>(), 1, 10));
        response.setPermission(Permission.ADMIN);

        when(groupsTabResponseBuilder.getGroupsTab(any())).thenReturn(response);

        mockMvc.perform(get("/groups?groupId=ROOT&paginatorPageSize=20&paginatorPage=1"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.breadcrumbs", notNullValue()))
                .andExpect(jsonPath("$.groupsPanel", notNullValue()))
                .andExpect(jsonPath("$.permission", is("ADMIN")));
    }

    @Test
    public void testAddGroupPaginated() throws Exception {

        AddGroupRequest request = new AddGroupRequest();
        request.setNewGroupName("New Group");
        request.setParentGroupId("abcd");
        request.setLeaf(true);
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(10);

        List<GroupNode> nodes = new ArrayList<>();
        PaginatedData<GroupNode> paginatedData = new PaginatedData<>(nodes, 1, 10);
        when(groupsTreeBuilder.listSubGroups(any(), any(), any())).thenReturn(paginatedData);

        mockMvc.perform(post("/group")
                .content(mapper.writeValueAsString(request))
                .accept(MediaType.APPLICATION_JSON)
                .contentType(MediaType.APPLICATION_JSON))
                .andExpect(status().isCreated());
    }

    @Test
    public void testDeleteGroup() throws Exception {

        mockMvc.perform(delete("/group/id?paginatorPageSize=20&paginatorPage=1&searchFilter="))
                .andDo(print())
                .andExpect(status().isOk());

        verify(groupsManager, times(1)).deleteGroup(eq("id"));
    }

    @Test
    public void testAddGroupTextPlain() throws Exception {

        GroupEntity peopleGroup = getPeopleGroup();

        when(groupsManager.getRoot()).thenReturn(getRoot());
        when(groupsManager.createGroup(any(GroupEntity.class), eq("people"), eq(false))).thenReturn(peopleGroup);
        when(groupsManager.createGroup(any(GroupEntity.class), eq("name.surname"), eq(true))).thenReturn(getNameSurnameGroup());

        mockMvc.perform(post("/group")
                .param("name", "people.name\\.surname")
                .param("leaf", "true")
                .accept(MediaType.TEXT_PLAIN)
                .contentType(MediaType.APPLICATION_FORM_URLENCODED))
                .andExpect(status().isCreated());

        verify(groupsService, times(2)).findGroupByParentAndName(any(GroupEntity.class), any());
        verify(groupsManager, times(1)).createGroup(argGroupIdEq(GroupsService.ROOT), eq("people"), eq(false));
        verify(groupsManager, times(1)).createGroup(argGroupIdEq("people_id"), eq("name.surname"), eq(true));
    }

    private GroupEntity argGroupIdEq(String groupId) {
        return argThat(g -> g.getId().equals(groupId));
    }

    private GroupEntity getRoot() {
        GroupEntity root = new GroupEntity();
        root.setId(GroupsService.ROOT);
        root.setName(GroupsService.ROOT);
        root.setPath("");
        return root;
    }

    private GroupEntity getPeopleGroup() {
        GroupEntity lbt = new GroupEntity();
        lbt.setId("people_id");
        lbt.setName("people");
        lbt.setPath("people_id");
        return lbt;
    }

    private GroupEntity getNameSurnameGroup() {
        GroupEntity inaf = new GroupEntity();
        inaf.setId("user_group_id");
        inaf.setName("name.surname");
        inaf.setPath("people_id.user_group_id");
        return inaf;
    }
}
