package it.inaf.ia2.gms.client;

import it.inaf.ia2.gms.client.model.Group;
import it.inaf.ia2.gms.client.model.Member;
import it.inaf.ia2.gms.client.model.Permission;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

public class GmsClient {

    private final String baseUrl;
    private final String authHeader;
    private RestTemplate restTemplate;

    public GmsClient(String baseUrl, String clientId, String clientSecret) {

        if (!baseUrl.endsWith("/")) {
            baseUrl += "/";
        }
        baseUrl += "ws/basic";

        this.baseUrl = baseUrl;

        String auth = clientId + ":" + clientSecret;
        byte[] encodedAuth = Base64.getEncoder().encode(
                auth.getBytes(StandardCharsets.UTF_8));
        authHeader = "Basic " + new String(encodedAuth);

        restTemplate = new RestTemplate();
    }

    /**
     * For testing purpose.
     */
    protected void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    public Group addGroup(List<String> names) {

        String url = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("group")
                .toUriString();

        HttpEntity<List<String>> httpEntity = getEntity(names);

        return restTemplate.exchange(url, HttpMethod.POST, httpEntity, Group.class).getBody();
    }

    public void removeGroup(List<String> names) {

        String url = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("group")
                .queryParam("names", names.toArray())
                .toUriString();

        restTemplate.exchange(url, HttpMethod.DELETE, getEntity(), Void.class);
    }

    public Member addMember(List<String> names, String userId) {

        String url = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("member")
                .toUriString();

        Map<String, Object> params = new HashMap<>();
        params.put("names", names);
        params.put("userId", userId);
        HttpEntity<Map<String, Object>> httpEntity = getEntity(params);

        return restTemplate.exchange(url, HttpMethod.POST, httpEntity, Member.class).getBody();
    }

    public void removeMember(List<String> names, String userId) {

        String url = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("member")
                .queryParam("names", names.toArray())
                .queryParam("userId", userId)
                .toUriString();

        restTemplate.exchange(url, HttpMethod.DELETE, getEntity(), Void.class);
    }

    public Permission addPermission(List<String> names, String userId, String permission) {

        String url = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("permission")
                .toUriString();

        Map<String, Object> params = new HashMap<>();
        params.put("names", names);
        params.put("userId", userId);
        params.put("permission", permission);
        HttpEntity<Map<String, Object>> httpEntity = getEntity(params);

        return restTemplate.exchange(url, HttpMethod.POST, httpEntity, Permission.class).getBody();
    }

    public void removePermission(List<String> names, String userId) {

        String url = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("permission")
                .queryParam("names", names.toArray())
                .queryParam("userId", userId)
                .toUriString();

        restTemplate.exchange(url, HttpMethod.DELETE, getEntity(), Void.class);
    }

    public void prepareToJoin(String fromUserId, String toUserId) {

        String url = UriComponentsBuilder.fromHttpUrl(baseUrl)
                .pathSegment("prepare-join")
                .toUriString();

        Map<String, Object> params = new HashMap<>();
        params.put("fromUserId", fromUserId);
        params.put("toUserId", toUserId);
        HttpEntity<Map<String, Object>> httpEntity = getEntity(params);

        restTemplate.exchange(url, HttpMethod.POST, httpEntity, Void.class);
    }

    private HttpEntity<?> getEntity() {
        return new HttpEntity<>(getHeaders());
    }

    private <T> HttpEntity<T> getEntity(T body) {
        return new HttpEntity<>(body, getHeaders());
    }

    private HttpHeaders getHeaders() {
        return new HttpHeaders() {
            {
                set("Authorization", authHeader);
            }
        };
    }
}
