package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.persistence.model.Group;
import it.inaf.ia2.gms.persistence.model.Membership;
import it.inaf.ia2.gms.persistence.model.User;
import it.inaf.ia2.gms.persistence.model.UserGroupPermission;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.Permission;
import java.util.List;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.test.context.junit4.SpringRunner;

@DataJpaTest
@AutoConfigureTestDatabase
@RunWith(SpringRunner.class)
public class NestedGroupsIntegrationTest {

    @Autowired
    private UsersRepository usersRepository;

    @Autowired
    private GroupsRepository groupsRepository;

    @Autowired
    private PermissionsRepository permissionsRepository;

    @Autowired
    private MembershipRepository membershipRepository;

    @Test
    public void testNestedGroupRetrieval() {

        GroupsService groupsService = new GroupsService(groupsRepository,
                permissionsRepository, membershipRepository);

        // Create user
        User user = new User();
        user.setId("user");
        user = usersRepository.save(user);

        // Test super admin
        UserGroupPermission superAdminPermission = new UserGroupPermission();
        superAdminPermission.setUser(user);
        superAdminPermission.setGroup(groupsRepository.findById(GroupsService.ROOT).get());
        superAdminPermission.setPermission(Permission.ADMIN);
        permissionsRepository.save(superAdminPermission);

        // Setup groups
        Group root = groupsService.getGroupById(GroupsService.ROOT);
        Group lbt = groupsService.addGroup(GroupsService.ROOT, "LBT", user);
        Group tng = groupsService.addGroup(GroupsService.ROOT, "TNG", user);
        Group radio = groupsService.addGroup(GroupsService.ROOT, "Radio", user);
        Group lbtInaf = groupsService.addGroup(lbt.getId(), "INAF", user);
        Group lbtInafProgram1 = groupsService.addGroup(lbtInaf.getId(), "P1", user);
        Group lbtInafProgram2 = groupsService.addGroup(lbtInaf.getId(), "P2", user);

        // Test super admin - level 0 (ROOT)
        List<GroupNode> groupNodes = groupNodes = groupsService.getSubgroups(root, user);
        assertEquals(3, groupNodes.size());
        GroupNode lbtGN = groupNodes.get(0);
        assertEquals("LBT", lbtGN.getGroupName());
        assertEquals(1, lbtGN.getPermissions().size());
        assertEquals(Permission.ADMIN, lbtGN.getPermissions().get(0));
        assertTrue(lbtGN.isHasChildren());
        GroupNode radioGN = groupNodes.get(1);
        assertEquals("Radio", radioGN.getGroupName());
        assertEquals(Permission.ADMIN, radioGN.getPermissions().get(0));
        assertFalse(radioGN.isHasChildren());
        GroupNode tngGN = groupNodes.get(2);
        assertEquals("TNG", tngGN.getGroupName());
        assertEquals(Permission.ADMIN, tngGN.getPermissions().get(0));
        assertFalse(tngGN.isHasChildren());

        // Test super admin - level 1
        groupNodes = groupsService.getSubgroups(lbt, user);
        assertEquals(1, groupNodes.size());
        GroupNode INAFGN = groupNodes.get(0);
        assertEquals("INAF", INAFGN.getGroupName());
        assertEquals(Permission.ADMIN, INAFGN.getPermissions().get(0));
        assertTrue(INAFGN.isHasChildren());

        // Test super admin - level 2
        groupNodes = groupsService.getSubgroups(lbtInaf, user);
        assertEquals(2, groupNodes.size());
        GroupNode p1 = groupNodes.get(0);
        assertEquals("P1", p1.getGroupName());
        assertEquals(1, p1.getPermissions().size());
        assertEquals(Permission.ADMIN, p1.getPermissions().get(0));
        assertFalse(p1.isHasChildren());
        GroupNode p2 = groupNodes.get(1);
        assertEquals("P2", p2.getGroupName());
        assertEquals(Permission.ADMIN, p2.getPermissions().get(0));
        assertFalse(p2.isHasChildren());

        // Setup lower permissions
        permissionsRepository.delete(superAdminPermission);

        UserGroupPermission p1Permission = new UserGroupPermission();
        p1Permission.setUser(user);
        p1Permission.setGroup(lbtInafProgram1);
        p1Permission.setPermission(Permission.MANAGE_MEMBERS);
        permissionsRepository.save(p1Permission);

        UserGroupPermission lbtPermission = new UserGroupPermission();
        lbtPermission.setUser(user);
        lbtPermission.setGroup(lbtInaf);
        lbtPermission.setPermission(Permission.VIEW_MEMBERS);
        permissionsRepository.save(lbtPermission);

        // Setup membership
        Membership membership = new Membership();
        membership.setUser(user);
        membership.setGroup(radio);
        membershipRepository.save(membership);

        // Check level 0 (ROOT)
        groupNodes = groupsService.getSubgroups(root, user);
        assertEquals(2, groupNodes.size());
        lbtGN = groupNodes.get(0);
        assertEquals("LBT", lbtGN.getGroupName());
        assertEquals(Permission.TRAVERSE, lbtGN.getPermissions().get(0));
        assertTrue(lbtGN.isHasChildren());
        radioGN = groupNodes.get(1);
        assertEquals("Radio", radioGN.getGroupName());
        assertEquals(Permission.TRAVERSE, radioGN.getPermissions().get(0));
        assertFalse(radioGN.isHasChildren());

        // Check level 1
        groupNodes = groupsService.getSubgroups(lbt, user);
        assertEquals(1, groupNodes.size());
        INAFGN = groupNodes.get(0);
        assertEquals("INAF", INAFGN.getGroupName());
        assertEquals(Permission.VIEW_MEMBERS, INAFGN.getPermissions().get(0));
        assertTrue(INAFGN.isHasChildren());

        // Check level 2
        groupNodes = groupsService.getSubgroups(lbtInaf, user);
        assertEquals(2, groupNodes.size());
        p1 = groupNodes.get(0);
        assertEquals("P1", p1.getGroupName());
        assertEquals(1, p1.getPermissions().size());
        assertEquals(Permission.MANAGE_MEMBERS, p1.getPermissions().get(0));
        assertFalse(p1.isHasChildren());
        p2 = groupNodes.get(1);
        assertEquals("P2", p2.getGroupName());
        assertEquals(Permission.VIEW_MEMBERS, p2.getPermissions().get(0));
        assertFalse(p2.isHasChildren());
    }
}
