package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.model.CreateGroupRequest;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.GroupsModelRequest;
import it.inaf.ia2.gms.model.GroupsModelResponse;
import it.inaf.ia2.gms.model.PaginatedData;
import it.inaf.ia2.gms.model.PaginatedModelRequest;
import it.inaf.ia2.gms.model.RenameGroupRequest;
import it.inaf.ia2.gms.persistence.model.Group;
import it.inaf.ia2.gms.persistence.model.User;
import it.inaf.ia2.gms.service.GroupsModelService;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.UsersService;
import java.util.List;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class GroupsController {

    @Autowired
    private SessionData session;

    @Autowired
    private UsersService usersService;

    @Autowired
    private GroupsModelService groupsModelService;

    @Autowired
    private GroupsService groupsService;

    @GetMapping(value = "/groups", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public GroupsModelResponse getGroupsModelResponse(@Valid GroupsModelRequest groupsModelRequest) {
        return groupsModelService.getGroupsModel(groupsModelRequest, session.getUserId());
    }

    @PostMapping(value = "/group", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> createGroup(@Valid @RequestBody CreateGroupRequest request) {

        User user = getUser();
        Group newGroup = groupsService.addGroup(request.getParentGroupId(), request.getNewGroupName(), user);

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(newGroup.getParentGroup(), request, user);

        return new ResponseEntity<>(groupsPanel, HttpStatus.CREATED);
    }

    @PutMapping(value = "/group/{groupId}", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> renameGroup(@PathVariable("groupId") String groupId, @Valid @RequestBody RenameGroupRequest request) {

        User user = getUser();
        Group renamedGroup = groupsService.renameGroup(groupId, request.getNewGroupName(), user);

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(renamedGroup.getParentGroup(), request, user);

        return ResponseEntity.ok(groupsPanel);
    }

    @DeleteMapping(value = "/group/{groupId}", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<?> deleteGroup(@PathVariable("groupId") String groupId, PaginatedModelRequest request) {

        User user = getUser();
        Group parent = groupsService.deleteGroup(groupId, user);

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent, request, user);

        return ResponseEntity.ok(groupsPanel);
    }

    private User getUser() {
        return usersService.getUserById(session.getUserId());
    }

    private PaginatedData<GroupNode> getGroupsPanel(Group parentGroup, PaginatedModelRequest paginatedRequest, User user) {
        List<GroupNode> groupNodes = groupsService.getSubgroups(parentGroup, user);
        return new PaginatedData<>(groupNodes, paginatedRequest.getPaginatorPage(), paginatedRequest.getPaginatorPageSize());
    }
}
