package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * Utility class for retrieving the complete names (including parents) from a
 * set of group paths.
 */
@Service
public class GroupNameService {

    @Autowired
    private GroupsDAO groupsDAO;

    public List<String> getGroupsNamesFromIdentifiers(Set<String> groupIdentifiers) {
        return getGroupsNames(groupsDAO.findGroupsByIds(groupIdentifiers));
    }

    /**
     * Returns the list of the group complete names, given a list of GroupEntity
     * objects.
     */
    public List<String> getGroupsNames(List<GroupEntity> groups) {

        // We need to return the complete group name, so it is necessary to load
        // all the parents too.
        Map<String, String> idNameMap = new HashMap<>();
        Set<String> allIdentifiers = getAllIdentifiers(groups);
        for (GroupEntity group : groupsDAO.findGroupsByIds(allIdentifiers)) {
            idNameMap.put(group.getId(), group.getName());
        }

        List<String> names = new ArrayList<>();
        for (GroupEntity group : groups) {
            names.add(getGroupCompleteName(group, idNameMap));
        }
        return names;
    }

    private Set<String> getAllIdentifiers(List<GroupEntity> groups) {

        Set<String> allIdentifiers = new HashSet<>();
        for (GroupEntity group : groups) {
            if (!"".equals(group.getPath())) {
                String[] ids = group.getPath().split("\\.");
                for (String id : ids) {
                    allIdentifiers.add(id);
                }
            }
        }

        return allIdentifiers;
    }

    private String getGroupCompleteName(GroupEntity group, Map<String, String> idNameMap) {

        if ("ROOT".equals(group.getId())) {
            return group.getName();
        }

        List<String> names = new ArrayList<>();

        for (String groupId : group.getPath().split("\\.")) {

            String groupName = idNameMap.get(groupId);

            // Dot inside names is considered a special character (because it is
            // used to separate the group from its parents), so we use a
            // backslash to escape it (client apps need to be aware of this).
            groupName = groupName.replace("\\.", "\\\\.");

            names.add(groupName);
        }

        return String.join(".", names);
    }

    /**
     * @param groupsIdPath map having group id as keys and group paths as values
     * @return map having group id as keys and group names as values
     */
    public Map<String, List<String>> getNames(List<Map.Entry<String, String>> groupsIdPath) {

        Set<String> allIdentifiers = new HashSet<>();
        for (Map.Entry<String, String> entry : groupsIdPath) {
            allIdentifiers.addAll(getIdentifiers(entry.getValue()));
        }

        Map<String, String> groupSingleNamesMap = getGroupSingleNamesMap(allIdentifiers);

        Map<String, List<String>> groupCompleteNamesMap = new HashMap<>();
        for (Map.Entry<String, String> entry : groupsIdPath) {
            List<String> groupCompleteName = getGroupCompleteName(groupSingleNamesMap, entry.getValue());
            groupCompleteNamesMap.put(entry.getKey(), groupCompleteName);
        }

        return groupCompleteNamesMap;
    }

    private Map<String, String> getGroupSingleNamesMap(Set<String> allIdentifiers) {
        Map<String, String> groupNamesMap = new HashMap<>();
        for (GroupEntity group : groupsDAO.findGroupsByIds(allIdentifiers)) {
            groupNamesMap.put(group.getId(), group.getName());
        }

        return groupNamesMap;
    }

    private List<String> getGroupCompleteName(Map<String, String> groupNamesMap, String groupPath) {
        List<String> names = new ArrayList<>();
        if (groupPath.isEmpty()) {
            names.add("Root");
        } else {
            List<String> identifiers = getIdentifiers(groupPath);
            for (String groupId : identifiers) {
                names.add(groupNamesMap.get(groupId));
            }
        }
        return names;
    }

    /**
     * Returns the list of all identifiers including parent ones.
     */
    private List<String> getIdentifiers(String groupPath) {
        List<String> identifiers = new ArrayList<>();
        if (!groupPath.isEmpty()) {
            for (String id : groupPath.split(Pattern.quote("."))) {
                identifiers.add(id);
            }
        }
        return identifiers;
    }
}
