package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.model.request.AddGroupRequest;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.request.DeleteGroupRequest;
import it.inaf.ia2.gms.model.request.GroupsRequest;
import it.inaf.ia2.gms.model.request.RenameGroupRequest;
import it.inaf.ia2.gms.model.request.SearchFilterRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsTreeBuilder;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class GroupsController {

    @Autowired
    private SessionData session;

    @Autowired
    private GroupsManager groupsManager;

    @Autowired
    private GroupsTreeBuilder groupsTreeBuilder;

    @Autowired
    private GroupsTabResponseBuilder groupsTabResponseBuilder;

    @GetMapping(value = "/groups", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<?> getGroupsTab(@Valid GroupsRequest request) {
        if (request.isOnlyPanel()) {
            // Only groupsPanel
            GroupEntity group = groupsManager.getGroupById(request.getGroupId());
            return ResponseEntity.ok(getGroupsPanel(group, request));
        } else {
            // Complete GroupsTabResponse
            return ResponseEntity.ok(groupsTabResponseBuilder.getGroupsTab(request));
        }
    }

    @PostMapping(value = "/group", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> createGroup(@Valid @RequestBody AddGroupRequest request) {

        GroupEntity parent = groupsManager.getGroupById(request.getParentGroupId());

        groupsManager.createGroup(parent, request.getNewGroupName(), request.isLeaf());

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent, request);

        return new ResponseEntity<>(groupsPanel, HttpStatus.CREATED);
    }

    @PutMapping(value = "/group/{groupId}", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<GroupNode>> updateGroup(@PathVariable("groupId") String groupId, @Valid @RequestBody RenameGroupRequest request) {

        GroupEntity updatedGroup = groupsManager.updateGroup(groupId, request.getNewGroupName(), request.isLeaf());

        GroupEntity parent = groupsManager.getGroupByPath(updatedGroup.getParentPath());

        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent, request);

        return ResponseEntity.ok(groupsPanel);
    }

    @DeleteMapping(value = "/group/{groupId}", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<?> deleteGroup(@PathVariable("groupId") String groupId, DeleteGroupRequest request) {

        GroupEntity parent = groupsManager.deleteGroup(groupId);
        PaginatedData<GroupNode> groupsPanel = getGroupsPanel(parent, request);

        return ResponseEntity.ok(groupsPanel);
    }

    private <T extends PaginatedModelRequest & SearchFilterRequest> PaginatedData<GroupNode> getGroupsPanel(GroupEntity parentGroup, T request) {
        return groupsTreeBuilder.listSubGroups(parentGroup, request, session.getUserId());
    }
}
