package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.request.AddPermissionRequest;
import it.inaf.ia2.gms.model.request.MemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.UserPermission;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class PermissionsController {

    @Autowired
    private SessionData session;

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private PermissionsService permissionsService;

    @Autowired
    private LoggingDAO loggingDAO;

    @GetMapping(value = "/permissions", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> getPermissionsTab(TabRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        Permission currentNodePermissions = permissionsService.getUserPermissionForGroup(group, session.getUserId());

        if (currentNodePermissions != Permission.ADMIN) {
            loggingDAO.logAction("Unauthorized attempt to see permissions");
            throw new UnauthorizedException("Only administrators can see the permissions");
        }

        List<UserPermission> permissions = permissionsService.getAllPermissions(group);
        PaginatedData<UserPermission> permissionsPanel = new PaginatedData<>(permissions, request.getPaginatorPage(), request.getPaginatorPageSize());

        return ResponseEntity.ok(permissionsPanel);
    }

    @GetMapping(value = "/permission", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<Map<String, Permission>> getUserPermission(@RequestParam("groupId") String groupId, @RequestParam("userId") String userId) {

        GroupEntity group = groupsService.getGroupById(groupId);
        permissionsService.verifyUserCanManagePermissions(group, session.getUserId());

        Permission permission = permissionsService.getUserPermissionForGroup(group, userId);
        if (permission == null) {
            return ResponseEntity.notFound().build();
        } else {
            Map<String, Permission> response = new HashMap<>();
            response.put("permission", permission);
            return ResponseEntity.ok(response);
        }
    }

    @PostMapping(value = "/permission", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> addPermission(@Valid @RequestBody AddPermissionRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        permissionsService.verifyUserCanManagePermissions(group, session.getUserId());

        permissionsService.addPermission(group, request.getUserId(), request.getPermission());
        loggingDAO.logAction("Permission added, group_id=" + request.getGroupId() + ", user_id="
                + request.getUserId() + ", permission=" + request.getPermission());

        return new ResponseEntity<>(getPermissionsPanel(group, request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/permission", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> deletePermission(@Valid MemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());
        permissionsService.verifyUserCanManagePermissions(group, session.getUserId());

        permissionsService.removePermission(group, request.getUserId());
        loggingDAO.logAction("Permission removed, group_id=" + request.getGroupId() + ", user_id=" + request.getUserId());

        return ResponseEntity.ok(getPermissionsPanel(group, request));
    }

    private PaginatedData<UserPermission> getPermissionsPanel(GroupEntity group, PaginatedModelRequest request) {
        List<UserPermission> permissions = permissionsService.getAllPermissions(group);
        return new PaginatedData<>(permissions, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
