package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.request.AddPermissionRequest;
import it.inaf.ia2.gms.model.request.MemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.UserPermission;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class PermissionsController {

    @Autowired
    private GroupsManager groupsManager;

    @Autowired
    private PermissionsManager permissionsManager;

    @GetMapping(value = "/permissions", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> getPermissionsTab(TabRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        List<UserPermission> permissions = permissionsManager.getAllPermissions(group);
        PaginatedData<UserPermission> permissionsPanel = new PaginatedData<>(permissions, request.getPaginatorPage(), request.getPaginatorPageSize());

        return ResponseEntity.ok(permissionsPanel);
    }

    @GetMapping(value = "/permission", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<Map<String, Permission>> getUserPermission(@RequestParam("groupId") String groupId, @RequestParam("userId") String userId) {

        GroupEntity group = groupsManager.getGroupById(groupId);

        Permission permission = permissionsManager.getUserPermission(group, userId);
        if (permission == null) {
            return ResponseEntity.notFound().build();
        } else {
            Map<String, Permission> response = new HashMap<>();
            response.put("permission", permission);
            return ResponseEntity.ok(response);
        }
    }

    @PostMapping(value = "/permission", consumes = MediaType.APPLICATION_JSON_UTF8_VALUE, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> addPermission(@Valid @RequestBody AddPermissionRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        permissionsManager.addPermission(group, request.getUserId(), request.getPermission());

        return new ResponseEntity<>(getPermissionsPanel(group, request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/permission", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<PaginatedData<UserPermission>> deletePermission(@Valid MemberRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        permissionsManager.removePermission(group, request.getUserId());

        return ResponseEntity.ok(getPermissionsPanel(group, request));
    }

    private PaginatedData<UserPermission> getPermissionsPanel(GroupEntity group, PaginatedModelRequest request) {
        List<UserPermission> permissions = permissionsManager.getAllPermissions(group);
        return new PaginatedData<>(permissions, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
