package it.inaf.ia2.gms.client;

import it.inaf.ia2.gms.client.call.HttpClientWrapper;
import it.inaf.ia2.gms.client.call.MockedHttpClientWrapper;
import it.inaf.ia2.gms.client.model.Permission;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import static org.junit.Assert.assertEquals;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentMatcher;
import org.mockito.ArgumentMatchers;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class GmsClientTest {

    private static final String BASE_URL = "http://base-url";

    private HttpClient httpClient;
    private GmsClient client;

    @Before
    public void setUp() {

        httpClient = mock(HttpClient.class);

        HttpClientWrapper clientWrapper = new MockedHttpClientWrapper(BASE_URL, httpClient);

        client = new GmsClient(BASE_URL);
        client.httpClientWrapper = clientWrapper;
    }

    @Test
    public void testGetMyGroups() {

        String body = "LBT.INAF\n"
                + "LBT.AZ";

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(200, body));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        List<String> groups = client.getMyGroups("LBT.");

        verify(httpClient, times(1)).sendAsync(endpointEq("GET", "search"), any());

        assertEquals(2, groups.size());
        assertEquals("INAF", groups.get(0));
        assertEquals("AZ", groups.get(1));
    }

    @Test
    public void testListGroups() {

        String body = "INAF\n"
                + "AZ";

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(200, body));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        List<String> groups = client.listGroups("LBT.");

        verify(httpClient, times(1)).sendAsync(endpointEq("GET", "list/LBT."), any());

        assertEquals(2, groups.size());
        assertEquals("INAF", groups.get(0));
        assertEquals("AZ", groups.get(1));
    }

    @Test
    public void testCreateGroup() {

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(201));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        client.createGroup("LBT.INAF", false);

        verify(httpClient, times(1)).sendAsync(endpointEq("POST", "LBT.INAF"), any());
    }

    @Test
    public void testDeleteGroup() {

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(204));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        client.deleteGroup("LBT.INAF");

        verify(httpClient, times(1)).sendAsync(endpointEq("DELETE", "LBT.INAF"), any());
    }

    @Test
    public void testAddMember() {

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(200));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        client.addMember("LBT.INAF", "user");

        verify(httpClient, times(1)).sendAsync(endpointEq("POST", "membership/LBT.INAF"), any());
    }

    @Test
    public void testRemoveMember() {

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(204));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        client.removeMember("LBT.INAF", "user");

        verify(httpClient, times(1)).sendAsync(endpointEq("DELETE", "membership/LBT.INAF?user_id=user"), any());
    }

    @Test
    public void testAddPermission() {

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(200));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        client.addPermission("LBT.INAF", "user", Permission.ADMIN);

        verify(httpClient, times(1)).sendAsync(endpointEq("POST", "permission/LBT.INAF"), any());
    }

    @Test
    public void testRemovePermission() {

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(204));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        client.removePermission("LBT.INAF", "user");

        verify(httpClient, times(1)).sendAsync(endpointEq("DELETE", "permission/LBT.INAF?user_id=user"), any());
    }

    private HttpResponse getMockedResponse(int statusCode, String body) {
        HttpResponse response = getMockedResponse(statusCode);
        InputStream in = new ByteArrayInputStream(body.getBytes(StandardCharsets.UTF_8));
        when(response.body()).thenReturn(in);
        return response;
    }

    private HttpResponse getMockedResponse(int statusCode) {
        HttpResponse response = mock(HttpResponse.class);
        when(response.statusCode()).thenReturn(statusCode);
        return response;
    }

    private HttpRequest endpointEq(String expectedMethod, String expectedEndpoint) {
        return ArgumentMatchers.argThat(endpointEqArgumentMatcher(expectedMethod, expectedEndpoint));
    }

    private ArgumentMatcher<HttpRequest> endpointEqArgumentMatcher(String expectedMethod, String expectedEndpoint) {

        return new ArgumentMatcher<HttpRequest>() {

            private final String expectedUri = BASE_URL + "/" + expectedEndpoint;

            @Override
            public boolean matches(HttpRequest request) {
                return expectedMethod.equals(request.method()) && expectedUri.equals(request.uri().toString());
            }

            @Override
            public String toString() {
                return expectedMethod + " " + expectedUri;
            }
        };
    }
}
