package it.inaf.ia2.gms.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.request.AddGroupRequest;
import it.inaf.ia2.gms.model.response.GroupsTabResponse;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.GroupsTreeBuilder;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.ArrayList;
import java.util.List;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

@RunWith(MockitoJUnitRunner.class)
public class GroupsControllerTest {

    @Mock
    private GroupsTabResponseBuilder groupsTabResponseBuilder;

    @Mock
    private GroupsService groupsService;

    @Mock
    private SessionData session;

    @Mock
    private PermissionsService permissionsService;

    @Mock
    private GroupsTreeBuilder groupsTreeBuilder;

    @InjectMocks
    private GroupsController controller;

    private final ObjectMapper mapper = new ObjectMapper();

    private MockMvc mockMvc;

    @Before
    public void init() {
        mockMvc = MockMvcBuilders.standaloneSetup(controller).build();
    }

    @Test
    public void testGetGroups() throws Exception {

        GroupsTabResponse response = new GroupsTabResponse();
        response.setBreadcrumbs(new ArrayList<>());
        response.setGroupsPanel(new PaginatedData<>(new ArrayList<>(), 1, 10));
        response.setPermission(Permission.ADMIN);

        when(groupsTabResponseBuilder.getGroupsTab(any())).thenReturn(response);

        mockMvc.perform(get("/groups?groupId=ROOT&paginatorPageSize=20&paginatorPage=1"))
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.breadcrumbs", notNullValue()))
                .andExpect(jsonPath("$.groupsPanel", notNullValue()))
                .andExpect(jsonPath("$.permission", is("ADMIN")));
    }

    @Test
    public void testAddGroups() throws Exception {

        AddGroupRequest request = new AddGroupRequest();
        request.setNewGroupName("New Group");
        request.setParentGroupId("abcd");
        request.setLeaf(true);
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(10);

        GroupEntity parent = new GroupEntity();
        when(groupsService.getGroupById(any())).thenReturn(parent);

        when(permissionsService.getUserPermissionForGroup(any(), any())).thenReturn(Permission.ADMIN);

        List<GroupNode> nodes = new ArrayList<>();
        PaginatedData<GroupNode> paginatedData = new PaginatedData<>(nodes, 1, 10);
        when(groupsTreeBuilder.listSubGroups(any(), any(), any())).thenReturn(paginatedData);

        when(session.getUserId()).thenReturn("admin_id");

        mockMvc.perform(post("/group")
                .content(mapper.writeValueAsString(request))
                .contentType(MediaType.APPLICATION_JSON_UTF8))
                .andExpect(status().isCreated());
    }
}
