package it.inaf.ia2.gms.authn;

import java.io.IOException;
import java.security.Principal;
import java.util.Map;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.oauth2.common.OAuth2AccessToken;
import org.springframework.security.oauth2.provider.token.store.jwk.JwkTokenStore;

public class JWTFilter implements Filter {

    private final JwkTokenStore jwkTokenStore;

    public JWTFilter(JwkTokenStore jwkTokenStore) {
        this.jwkTokenStore = jwkTokenStore;
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain fc) throws IOException, ServletException {

        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

        String authHeader = request.getHeader("Authorization");
        if (authHeader == null) {
            response.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Missing Authorization token");
            return;
        }

        authHeader = authHeader.replace("Bearer", "").trim();

        OAuth2AccessToken accessToken = jwkTokenStore.readAccessToken(authHeader);
        if (accessToken.isExpired()) {
            response.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Access token is expired");
            return;
        }

        Map<String, Object> claims = accessToken.getAdditionalInformation();

        String principal = (String) claims.get("sub");
        if (principal == null) {
            response.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Invalid access token: missing sub claim");
            return;
        }

        ServletRequest wrappedRequest = new ServletRequestWithJWTPrincipal(request, principal);

        fc.doFilter(wrappedRequest, res);
    }

    private static class ServletRequestWithJWTPrincipal extends HttpServletRequestWrapper {

        private final String principal;

        public ServletRequestWithJWTPrincipal(HttpServletRequest request, String principal) {
            super(request);
            this.principal = principal;
        }

        @Override
        public Principal getUserPrincipal() {
            return new UsernamePasswordAuthenticationToken(principal, null);
        }
    }
}
