package it.inaf.ia2.gms.client;

import it.inaf.ia2.gms.client.model.Group;
import it.inaf.ia2.gms.client.model.Member;
import it.inaf.ia2.gms.client.model.Permission;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import static org.junit.Assert.assertEquals;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentCaptor;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

@RunWith(MockitoJUnitRunner.class)
public class GmsClientTest {

    private static final String BASE_URL = "http://base-url";

    private RestTemplate restTemplate;
    private GmsClient client;

    @Before
    public void setUp() {

        restTemplate = mock(RestTemplate.class);

        client = new GmsClient(BASE_URL, "test", "test");
        client.setRestTemplate(restTemplate);

        when(restTemplate.exchange(anyString(), any(HttpMethod.class), any(HttpEntity.class), any(Class.class)))
                .thenReturn(new ResponseEntity<>(HttpStatus.I_AM_A_TEAPOT));
    }

    @Test
    public void testCreateGroup() {

        List<String> names = Arrays.asList("LBT", "INAF");

        client.addGroup(names);

        ArgumentCaptor<HttpEntity> entityCaptor = ArgumentCaptor.forClass(HttpEntity.class);
        verify(restTemplate, times(1)).exchange(eq(BASE_URL + "/ws/group"),
                eq(HttpMethod.POST), entityCaptor.capture(), eq(Group.class));

        HttpEntity<?> entity = entityCaptor.getValue();
        verifyAuthHeaders(entity);
        verifyBody(entity, names);
    }

    @Test
    public void testDeleteGroup() {

        List<String> names = Arrays.asList("LBT", "INAF");

        client.removeGroup(names);

        ArgumentCaptor<HttpEntity> entityCaptor = ArgumentCaptor.forClass(HttpEntity.class);
        verify(restTemplate, times(1)).exchange(eq(BASE_URL + "/ws/group?names=LBT&names=INAF"),
                eq(HttpMethod.DELETE), entityCaptor.capture(), eq(Void.class));

        HttpEntity<?> entity = entityCaptor.getValue();
        verifyAuthHeaders(entity);
    }

    @Test
    public void testAddMember() {

        List<String> names = Arrays.asList("LBT", "INAF");

        client.addMember(names, "user_id");

        ArgumentCaptor<HttpEntity> entityCaptor = ArgumentCaptor.forClass(HttpEntity.class);
        verify(restTemplate, times(1)).exchange(eq(BASE_URL + "/ws/member"),
                eq(HttpMethod.POST), entityCaptor.capture(), eq(Member.class));

        HttpEntity<?> entity = entityCaptor.getValue();
        verifyAuthHeaders(entity);

        Map<String, Object> expectedBody = new HashMap<>();
        expectedBody.put("names", names);
        expectedBody.put("userId", "user_id");

        verifyBody(entity, expectedBody);
    }

    @Test
    public void testRemoveMember() {

        List<String> names = Arrays.asList("LBT", "INAF");

        client.removeMember(names, "user_id");

        ArgumentCaptor<HttpEntity> entityCaptor = ArgumentCaptor.forClass(HttpEntity.class);
        verify(restTemplate, times(1)).exchange(eq(BASE_URL + "/ws/member?names=LBT&names=INAF&userId=user_id"),
                eq(HttpMethod.DELETE), entityCaptor.capture(), eq(Void.class));

        HttpEntity<?> entity = entityCaptor.getValue();
        verifyAuthHeaders(entity);
    }

    @Test
    public void testAddPermission() {

        List<String> names = Arrays.asList("LBT", "INAF");

        client.addPermission(names, "user_id", "ADMIN");

        ArgumentCaptor<HttpEntity> entityCaptor = ArgumentCaptor.forClass(HttpEntity.class);
        verify(restTemplate, times(1)).exchange(eq(BASE_URL + "/ws/permission"),
                eq(HttpMethod.POST), entityCaptor.capture(), eq(Permission.class));

        HttpEntity<?> entity = entityCaptor.getValue();
        verifyAuthHeaders(entity);

        Map<String, Object> expectedBody = new HashMap<>();
        expectedBody.put("names", names);
        expectedBody.put("userId", "user_id");
        expectedBody.put("permission", "ADMIN");

        verifyBody(entity, expectedBody);
    }

    @Test
    public void testDeletePermission() {

        List<String> names = Arrays.asList("LBT", "INAF");

        client.removePermission(names, "user_id");

        ArgumentCaptor<HttpEntity> entityCaptor = ArgumentCaptor.forClass(HttpEntity.class);
        verify(restTemplate, times(1)).exchange(eq(BASE_URL + "/ws/permission?names=LBT&names=INAF&userId=user_id"),
                eq(HttpMethod.DELETE), entityCaptor.capture(), eq(Void.class));

        HttpEntity<?> entity = entityCaptor.getValue();
        verifyAuthHeaders(entity);
    }

    @Test
    public void testPrepareToJoin() {

        String fromUserId = "from_user_id";
        String toUserId = "to_user_id";

        client.prepareToJoin(fromUserId, toUserId);

        ArgumentCaptor<HttpEntity> entityCaptor = ArgumentCaptor.forClass(HttpEntity.class);
        verify(restTemplate, times(1)).exchange(eq(BASE_URL + "/ws/prepare-join"),
                eq(HttpMethod.POST), entityCaptor.capture(), eq(Void.class));

        HttpEntity<?> entity = entityCaptor.getValue();
        verifyAuthHeaders(entity);

        Map<String, Object> expectedBody = new HashMap<>();
        expectedBody.put("fromUserId", fromUserId);
        expectedBody.put("toUserId", toUserId);

        verifyBody(entity, expectedBody);
    }

    private void verifyAuthHeaders(HttpEntity<?> entity) {//
        String authHeader = entity.getHeaders().getFirst("Authorization");
        assertEquals("Basic dGVzdDp0ZXN0", authHeader);
    }

    private <T> void verifyBody(HttpEntity<?> entity, T body) {
        assertEquals(entity.getBody(), body);
    }
}
