package it.inaf.ia2.gms.manager;

import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.PermissionsService;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class PermissionsManagerTest {

    private static final String USER_ID = "USER_ID";
    private static final String TARGET_USER_ID = "TARGET_USER_ID";

    @Mock
    private PermissionsService permissionsService;
    @Mock
    private HttpServletRequest request;
    @Mock
    private LoggingDAO loggingDAO;

    @InjectMocks
    private PermissionsManager permissionsManager;

    @Before
    public void setUp() {
        UserAwareComponentTestUtil.setUser(permissionsManager, USER_ID);
    }

    @Test
    public void addPermissionFromManager() {

        GroupEntity group = getGroup();

        when(permissionsService.findUserPermissions(eq(group), eq(USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.MANAGE_MEMBERS));

        permissionsManager.addPermission(group, TARGET_USER_ID, Permission.VIEW_MEMBERS);

        verify(permissionsService, times(1)).addPermission(eq(group), eq(TARGET_USER_ID), eq(Permission.VIEW_MEMBERS));
    }

    @Test
    public void addPermissionFromAdmin() {

        GroupEntity group = getGroup();

        when(permissionsService.findUserPermissions(eq(group), eq(USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.ADMIN));

        permissionsManager.addPermission(group, TARGET_USER_ID, Permission.MANAGE_MEMBERS);

        verify(permissionsService, times(1)).addPermission(eq(group), eq(TARGET_USER_ID), eq(Permission.MANAGE_MEMBERS));
    }

    @Test(expected = UnauthorizedException.class)
    public void addPermissionFromUnauthorized() {

        GroupEntity group = getGroup();

        when(permissionsService.findUserPermissions(eq(group), eq(USER_ID)))
                .thenReturn(new ArrayList<>());

        permissionsManager.addPermission(group, TARGET_USER_ID, Permission.MANAGE_MEMBERS);
    }

    @Test
    public void removePermissionFromManager() {

        GroupEntity group = getGroup();

        when(permissionsService.findUserPermissions(eq(group), eq(USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.MANAGE_MEMBERS));

        when(permissionsService.findUserPermissions(eq(group), eq(TARGET_USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.VIEW_MEMBERS));

        permissionsManager.removePermission(group, TARGET_USER_ID);

        verify(permissionsService, times(1)).removePermission(eq(group), eq(TARGET_USER_ID));
    }

    @Test
    public void removePermissionFromManagerAddedByAdmin() {

        GroupEntity group = getGroup();

        when(permissionsService.findUserPermissions(eq(group), eq(USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.MANAGE_MEMBERS));

        when(permissionsService.findUserPermissions(eq(group), eq(TARGET_USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.MANAGE_MEMBERS));

        permissionsManager.removePermission(group, TARGET_USER_ID);

        verify(permissionsService, never()).removePermission(any(), any());
    }

    @Test
    public void removePermissionFromAdmin() {

        GroupEntity group = getGroup();

        when(permissionsService.findUserPermissions(eq(group), eq(USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.ADMIN));

        permissionsManager.removePermission(group, TARGET_USER_ID);

        verify(permissionsService, times(1)).removePermission(eq(group), eq(TARGET_USER_ID));
    }

    @Test(expected = UnauthorizedException.class)
    public void removePermissionFromUnauthorized() {

        GroupEntity group = getGroup();

        when(permissionsService.findUserPermissions(eq(group), eq(USER_ID)))
                .thenReturn(getUserPermissions(group, Permission.VIEW_MEMBERS));

        permissionsManager.removePermission(group, TARGET_USER_ID);
    }

    private List<PermissionEntity> getUserPermissions(GroupEntity group, Permission permission) {
        PermissionEntity entity = new PermissionEntity();
        entity.setPermission(permission);
        entity.setGroupId(group.getId());
        entity.setGroupPath(group.getPath());
        entity.setUserId(USER_ID);
        return Collections.singletonList(entity);
    }

    private GroupEntity getGroup() {
        GroupEntity group = new GroupEntity();
        group.setId("group_id");
        group.setPath("parent_id.group_id");
        return group;
    }
}
