package it.inaf.ia2.gms.model.response;

import java.util.ArrayList;
import java.util.List;

public class PaginatedData<T> {

    // how many page index links have to be shown
    private static final int LINKS_SIZE = 5;

    private final List<T> items;
    private final int currentPage;
    private final List<Integer> links;
    private final int totalItems;
    private final int pageSize;
    private final int totalPages;
    private final boolean hasPreviousPages;
    private final boolean hasFollowingPages;

    public PaginatedData(List<T> allItems, int page, int pageSize) {

        if (page < 1) {
            throw new IllegalArgumentException("Page numbers start from 1");
        }

        totalItems = allItems.size();
        this.pageSize = pageSize;

        totalPages = totalItems == 0 ? 1 : (int) Math.ceil((double) totalItems / pageSize);

        if (page <= totalPages) {
            this.currentPage = page;
        } else {
            // back to last available page
            this.currentPage = totalPages;
        }

        if (allItems.isEmpty()) {
            items = allItems;
        } else {
            int firstElementIndex = (currentPage - 1) * pageSize;
            int lastElementIndex = Math.min(currentPage * pageSize - 1, totalItems - 1);
            items = new ArrayList<>(allItems.subList(firstElementIndex, lastElementIndex + 1));
        }

        links = new ArrayList<>();
        int start = currentPage - LINKS_SIZE / 2;
        if (start > 1 && start + LINKS_SIZE > totalPages) {
            start = totalPages - LINKS_SIZE + 1;
        }
        if (start < 1) {
            start = 1;
        }
        int i = start;
        hasPreviousPages = start > 1;
        while (i < start + LINKS_SIZE && i <= totalPages) {
            links.add(i++);
        }
        hasFollowingPages = totalPages > i;
    }

    public List<T> getItems() {
        return items;
    }

    public int getCurrentPage() {
        return currentPage;
    }

    public List<Integer> getLinks() {
        return links;
    }

    public int getTotalItems() {
        return totalItems;
    }

    public int getPageSize() {
        return pageSize;
    }

    public int getTotalPages() {
        return totalPages;
    }

    public boolean isHasPreviousPages() {
        return hasPreviousPages;
    }

    public boolean isHasFollowingPages() {
        return hasFollowingPages;
    }
}
