package it.inaf.ia2.gms;

import it.inaf.ia2.aa.AuthConfig;
import it.inaf.ia2.aa.ServiceLocator;
import it.inaf.ia2.aa.data.ServletCodeRequestData;
import it.inaf.ia2.client.QueryStringBuilder;
import it.inaf.ia2.client.UriCustomizer;
import static it.inaf.ia2.gms.authn.ClientDbFilter.CLIENT_DB;
import it.inaf.ia2.gms.exception.BadRequestException;
import it.inaf.ia2.rap.client.RapClient;
import java.net.URI;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.annotation.EnableTransactionManagement;

@SpringBootApplication
@Configuration
@EnableTransactionManagement
public class GmsApplication {

    public static void main(String[] args) {
        SpringApplication.run(GmsApplication.class, args);
    }

    @Bean
    public AuthConfig authConfig() {
        return ServiceLocator.getInstance().getConfig();
    }

    @Bean
    public RapClient rapClient(AuthConfig authConfig) {

        URI defaultAuthorizationUri = URI.create(authConfig.getRapBaseUri())
                .resolve(authConfig.getUserAuthorizationEndpoint());

        URI defaultAccessTokenUri = URI.create(authConfig.getRapBaseUri())
                .resolve(authConfig.getAccessTokenEndpoint());

        RapClient rapClient = ServiceLocator.getInstance().getRapClient();

        rapClient.setAuthorizationUriCustomizer(new UriCustomizer<HttpServletRequest>() {

            @Override
            public URI getBaseUri(HttpServletRequest req) {
                // for a better security we should check for allowed redirects
                String redirect = req.getParameter("redirect");
                if (redirect != null) {
                    return URI.create(redirect);
                }
                return defaultAuthorizationUri;
            }

            @Override
            public void customizeQueryString(HttpServletRequest req, QueryStringBuilder queryStringBuilder) {
                String clientDb = req.getParameter(CLIENT_DB);
                if (clientDb == null) {
                    HttpSession session = req.getSession(false);
                    if (session != null) {
                        clientDb = (String) session.getAttribute(CLIENT_DB);
                    }
                }
                if (clientDb == null) {
                    throw new BadRequestException("client_db not set");
                }
                queryStringBuilder.param(CLIENT_DB, clientDb);
            }
        });

        rapClient.setAccessTokenUriCustomizer(new UriCustomizer<ServletCodeRequestData>() {
            @Override
            public URI getBaseUri(ServletCodeRequestData req) {
                String redirect = req.getCodeRequest().getParameter("token_uri");
                if (redirect != null) {
                    return URI.create(redirect);
                }
                return defaultAccessTokenUri;
            }
        });

        return rapClient;
    }
}
