package it.inaf.ia2.gms.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import it.inaf.ia2.client.BaseClient;
import it.inaf.ia2.gms.client.call.AddInvitedRegistrationCall;
import it.inaf.ia2.gms.client.call.AddMemberCall;
import it.inaf.ia2.gms.client.call.AddPermissionCall;
import it.inaf.ia2.gms.client.call.CreateGroupCall;
import it.inaf.ia2.gms.client.call.DeleteGroupCall;
import it.inaf.ia2.gms.client.call.GetGroupPermissionsCall;
import it.inaf.ia2.gms.client.call.GetGroupStatusCall;
import it.inaf.ia2.gms.client.call.GetMemberEmailAddresses;
import it.inaf.ia2.gms.client.call.GetUserGroupsCall;
import it.inaf.ia2.gms.client.call.GetUserPermissionsCall;
import it.inaf.ia2.gms.client.call.ListGroupsCall;
import it.inaf.ia2.gms.client.call.RemoveMemberCall;
import it.inaf.ia2.gms.client.call.RemovePermissionCall;
import it.inaf.ia2.gms.client.call.SetPermissionCall;
import it.inaf.ia2.gms.client.model.GroupPermission;
import it.inaf.ia2.gms.client.model.Permission;
import it.inaf.ia2.gms.client.model.UserPermission;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;

public class GmsClient extends BaseClient {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    private String accessToken;

    public GmsClient(String gmsBaseUri) {
        super(gmsBaseUri);
    }

    public GmsClient setAccessToken(String accessToken) {
        this.accessToken = accessToken;
        return this;
    }

    public List<String> getMyGroups(String prefix) {
        return new GetUserGroupsCall(this).getUserGroups(prefix);
    }

    @Override
    public HttpRequest.Builder newRequest(String endpoint) {
        if (accessToken == null) {
            throw new IllegalStateException("Access token is null");
        }
        return super.newRequest(getUri(endpoint))
                .setHeader("Authorization", "Bearer " + accessToken);
    }

    @Override
    public HttpRequest.Builder newRequest(URI uri) {
        if (accessToken == null) {
            throw new IllegalStateException("Access token is null");
        }
        return super.newRequest(uri)
                .setHeader("Authorization", "Bearer " + accessToken);
    }

    @Override
    protected <T> String getInvalidStatusCodeExceptionMessage(HttpRequest request, HttpResponse<T> response) {
        return response.headers().firstValue("Content-Type")
                .map(contentType -> {
                    try {
                        if (contentType.startsWith("text/plain")) {
                            String errorResponseBody = null;
                            if (response.body() instanceof String) {
                                errorResponseBody = (String) response.body();
                            } else if (response.body() instanceof InputStream) {
                                errorResponseBody = new String(((InputStream) response.body()).readAllBytes(), StandardCharsets.UTF_8);
                            }
                            if (errorResponseBody != null && !errorResponseBody.isBlank()) {
                                return errorResponseBody;
                            }
                        } else if (contentType.startsWith("application/json")
                                || contentType.startsWith("text/json")) {
                            Map<String, Object> errorResponseBody = null;
                            if (response.body() instanceof String) {
                                errorResponseBody = MAPPER.readValue((String) response.body(), Map.class);
                            } else if (response.body() instanceof InputStream) {
                                errorResponseBody = MAPPER.readValue((InputStream) response.body(), Map.class);
                            }
                            if (errorResponseBody != null && errorResponseBody.containsKey("error")) {
                                return (String) errorResponseBody.get("error");
                            }
                        }
                    } catch (IOException ex) {
                    }
                    return null;
                }).orElse(super.getInvalidStatusCodeExceptionMessage(request, response));
    }

    public List<String> listGroups(String prefix) {
        return new ListGroupsCall(this).listGroups(prefix);
    }

    public List<String> getUserGroups(String userId) {
        return new GetUserGroupsCall(this).getUserGroups(userId);
    }

    public List<String> getUserGroups(String userId, String prefix) {
        return new GetUserGroupsCall(this).getUserGroups(userId, prefix);
    }

    public void createGroup(String completeGroupName, boolean leaf) {
        new CreateGroupCall(this).createGroup(completeGroupName, leaf);
    }

    public void deleteGroup(String completeGroupName) {
        new DeleteGroupCall(this).deleteGroup(completeGroupName);
    }

    public void addMember(String completeGroupName, String userId) {
        new AddMemberCall(this).addMember(completeGroupName, userId);
    }

    public void removeMember(String completeGroupName, String userId) {
        new RemoveMemberCall(this).removeMember(completeGroupName, userId);
    }

    public String addPermission(String completeGroupName, String userId, Permission permission) {
        return new AddPermissionCall(this).addPermission(completeGroupName, userId, permission);
    }

    public String setPermission(String completeGroupName, String userId, Permission permission) {
        return new SetPermissionCall(this).setPermission(completeGroupName, userId, permission);
    }

    public void removePermission(String completeGroupName, String userId) {
        new RemovePermissionCall(this).removePermission(completeGroupName, userId);
    }

    public List<UserPermission> getUserPermissions(String userId) {
        return new GetUserPermissionsCall(this).getUserPermissions(userId);
    }

    public List<GroupPermission> getGroupPermissions(String groupId) {
        return new GetGroupPermissionsCall(this).getGroupPermissions(groupId);
    }

    public void addInvitedRegistration(String token, String email, Map<String, Permission> groupsPermissions) {
        new AddInvitedRegistrationCall(this).addInvitedRegistration(token, email, groupsPermissions);
    }

    public List<String> getMemberEmailAddresses(String groupId, Permission permission) {
        return new GetMemberEmailAddresses(this).getMemberEmailAddresses(groupId, permission);
    }

    public List<String[]> getStatus(String groupCompleteName) {
        return new GetGroupStatusCall(this).getStatus(groupCompleteName);
    }
}
