package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.DataSourceConfig;
import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.manager.UserAwareComponentTestUtil;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.model.GroupNode;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.request.GroupsRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupsTreeBuilder;
import it.inaf.ia2.gms.service.PermissionsService;
import it.inaf.ia2.gms.authn.RapClient;
import java.util.List;
import javax.sql.DataSource;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringRunner;

@RunWith(SpringRunner.class)
@ContextConfiguration(classes = DataSourceConfig.class)
public class NestedGroupsIntegrationTest {

    @MockBean
    private LoggingDAO loggingDAO;

    @MockBean
    private RapClient rapClient;

    @Autowired
    private DataSource dataSource;

    @Test
    public void testNestedGroupRetrieval() {

        String userId = "USER_ID";

        GroupsDAO groupsDAO = new GroupsDAO(dataSource);
        PermissionsDAO permissionsDAO = new PermissionsDAO(dataSource);
        MembershipsDAO membershipsDAO = new MembershipsDAO(dataSource);
        InvitedRegistrationDAO invitedRegistrationDAO = new InvitedRegistrationDAO(dataSource);
        PermissionsService permissionsService = new PermissionsService(permissionsDAO, loggingDAO);
        PermissionsManager permissionsManager = new PermissionsManager(permissionsService, rapClient, loggingDAO);
        UserAwareComponentTestUtil.setUser(permissionsManager, userId);
        GroupsService groupsService = new GroupsService(groupsDAO, permissionsDAO, membershipsDAO, invitedRegistrationDAO, loggingDAO);
        GroupsManager groupsManager = new GroupsManager(groupsService, permissionsManager, loggingDAO);
        GroupsTreeBuilder groupsTreeBuilder = new GroupsTreeBuilder(groupsDAO, permissionsDAO);

        // Test super admin
        PermissionEntity superAdminPermission = new PermissionEntity();
        superAdminPermission.setUserId(userId);
        superAdminPermission.setGroupId(GroupsService.ROOT);
        superAdminPermission.setPermission(Permission.ADMIN);
        superAdminPermission.setGroupPath("");
        permissionsDAO.createOrUpdatePermission(superAdminPermission);

        // Setup groups
        GroupEntity root = groupsManager.getRoot();
        GroupEntity lbt = groupsService.addGroup(root, "LBT", false);
        GroupEntity tng = groupsService.addGroup(root, "TNG", false);
        GroupEntity radio = groupsService.addGroup(root, "Radio", false);
        GroupEntity lbtInaf = groupsService.addGroup(lbt, "INAF", false);
        GroupEntity lbtInafProgram1 = groupsService.addGroup(lbtInaf, "P1", false);
        GroupEntity lbtInafProgram2 = groupsService.addGroup(lbtInaf, "P2", false);

        GroupsRequest request = new GroupsRequest();
        request.setPaginatorPage(1);
        request.setPaginatorPageSize(50);

        // Test super admin - level 0 (ROOT)
        List<GroupNode> groupNodes = groupNodes = groupsTreeBuilder.listSubGroups(root, request, userId).getItems();
        assertEquals(3, groupNodes.size());
        GroupNode lbtGN = groupNodes.get(0);
        assertEquals("LBT", lbtGN.getGroupName());
        assertEquals(Permission.ADMIN, lbtGN.getPermission());
        assertTrue(lbtGN.isHasChildren());
        GroupNode radioGN = groupNodes.get(1);
        assertEquals("Radio", radioGN.getGroupName());
        assertEquals(Permission.ADMIN, radioGN.getPermission());
        assertFalse(radioGN.isHasChildren());
        GroupNode tngGN = groupNodes.get(2);
        assertEquals("TNG", tngGN.getGroupName());
        assertEquals(Permission.ADMIN, tngGN.getPermission());
        assertFalse(tngGN.isHasChildren());

        // Test super admin - level 1
        groupNodes = groupsTreeBuilder.listSubGroups(lbt, request, userId).getItems();
        assertEquals(1, groupNodes.size());
        GroupNode INAFGN = groupNodes.get(0);
        assertEquals("INAF", INAFGN.getGroupName());
        assertEquals(Permission.ADMIN, INAFGN.getPermission());
        assertTrue(INAFGN.isHasChildren());

        // Test super admin - level 2
        groupNodes = groupsTreeBuilder.listSubGroups(lbtInaf, request, userId).getItems();
        assertEquals(2, groupNodes.size());
        GroupNode p1 = groupNodes.get(0);
        assertEquals("P1", p1.getGroupName());
        assertEquals(Permission.ADMIN, p1.getPermission());
        assertFalse(p1.isHasChildren());
        GroupNode p2 = groupNodes.get(1);
        assertEquals("P2", p2.getGroupName());
        assertEquals(Permission.ADMIN, p2.getPermission());
        assertFalse(p2.isHasChildren());

        // Setup lower permissions
        permissionsDAO.deletePermission(superAdminPermission.getGroupId(), superAdminPermission.getUserId());

        PermissionEntity p1Permission = new PermissionEntity();
        p1Permission.setUserId(userId);
        p1Permission.setGroupId(lbtInafProgram1.getId());
        p1Permission.setPermission(Permission.MANAGE_MEMBERS);
        p1Permission.setGroupPath(lbtInafProgram1.getPath());
        permissionsDAO.createOrUpdatePermission(p1Permission);

        PermissionEntity lbtPermission = new PermissionEntity();
        lbtPermission.setUserId(userId);
        lbtPermission.setGroupId(lbtInaf.getId());
        lbtPermission.setPermission(Permission.VIEW_MEMBERS);
        lbtPermission.setGroupPath(lbtInaf.getPath());
        permissionsDAO.createOrUpdatePermission(lbtPermission);

        PermissionEntity radioPermission = new PermissionEntity();
        radioPermission.setUserId(userId);
        radioPermission.setGroupId(radio.getId());
        radioPermission.setPermission(Permission.VIEW_MEMBERS);
        radioPermission.setGroupPath(radio.getPath());
        permissionsDAO.createOrUpdatePermission(radioPermission);

        // Check level 0 (ROOT)
        groupNodes = groupsTreeBuilder.listSubGroups(root, request, userId).getItems();
        assertEquals(2, groupNodes.size());
        lbtGN = groupNodes.get(0);
        assertEquals("LBT", lbtGN.getGroupName());
        assertEquals(Permission.TRAVERSE, lbtGN.getPermission());
        assertTrue(lbtGN.isHasChildren());
        radioGN = groupNodes.get(1);
        assertEquals("Radio", radioGN.getGroupName());
        assertEquals(Permission.VIEW_MEMBERS, radioGN.getPermission());
        assertFalse(radioGN.isHasChildren());

        // Check level 1
        groupNodes = groupsTreeBuilder.listSubGroups(lbt, request, userId).getItems();
        assertEquals(1, groupNodes.size());
        INAFGN = groupNodes.get(0);
        assertEquals("INAF", INAFGN.getGroupName());
        assertEquals(Permission.VIEW_MEMBERS, INAFGN.getPermission());
        assertTrue(INAFGN.isHasChildren());

        // Check level 2
        groupNodes = groupsTreeBuilder.listSubGroups(lbtInaf, request, userId).getItems();
        assertEquals(2, groupNodes.size());
        p1 = groupNodes.get(0);
        assertEquals("P1", p1.getGroupName());
        assertEquals(Permission.MANAGE_MEMBERS, p1.getPermission());
        assertFalse(p1.isHasChildren());
        p2 = groupNodes.get(1);
        assertEquals("P2", p2.getGroupName());
        assertEquals(Permission.VIEW_MEMBERS, p2.getPermission());
        assertFalse(p2.isHasChildren());
    }
}
