package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.exception.BadRequestException;
import it.inaf.ia2.gms.manager.GroupsManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.request.AddPermissionRequest;
import it.inaf.ia2.gms.model.request.MemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.model.RapUserPermission;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.model.request.UpdatePermissionRequest;
import it.inaf.ia2.gms.model.response.UserPermission;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupNameService;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class PermissionsController {

    @Autowired
    private GroupsManager groupsManager;

    @Autowired
    private PermissionsManager permissionsManager;

    @GetMapping(value = "/permissions", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUserPermission>> getPermissionsTab(TabRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        PaginatedData<RapUserPermission> permissionsPanel = getPermissionsPanel(group, request);

        return ResponseEntity.ok(permissionsPanel);
    }

    @GetMapping(value = "/permission", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Map<String, Permission>> getUserPermission(@RequestParam("groupId") String groupId, @RequestParam("userId") String userId) {

        GroupEntity group = groupsManager.getGroupById(groupId);

        Permission permission = permissionsManager.getDirectUserPermission(group, userId);
        if (permission == null) {
            return ResponseEntity.noContent().build();
        } else {
            Map<String, Permission> response = new HashMap<>();
            response.put("permission", permission);
            return ResponseEntity.ok(response);
        }
    }

    @PostMapping(value = "/permission", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUserPermission>> addPermission(@Valid @RequestBody AddPermissionRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        if (request.isOverride()) {
            permissionsManager.updatePermission(group, request.getUserId(), request.getPermission());
        } else {
            permissionsManager.addPermission(group, request.getUserId(), request.getPermission());
        }

        return new ResponseEntity<>(getPermissionsPanel(group, request), HttpStatus.CREATED);
    }

    @PutMapping(value = "/permission", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<Map<String, String>> updatePermission(@Valid @RequestBody UpdatePermissionRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        PermissionEntity updatedEntity = permissionsManager.updatePermission(group, request.getUserId(), request.getPermission());

        Map<String, String> response = new HashMap<>();
        response.put("permission", updatedEntity.getPermission().toString());

        return ResponseEntity.ok(response);
    }

    @DeleteMapping(value = "/permission", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUserPermission>> deletePermission(@Valid MemberRequest request) {

        GroupEntity group = groupsManager.getGroupById(request.getGroupId());
        permissionsManager.removePermission(group, request.getUserId());

        return ResponseEntity.ok(getPermissionsPanel(group, request));
    }

    private PaginatedData<RapUserPermission> getPermissionsPanel(GroupEntity group, PaginatedModelRequest request) {
        List<RapUserPermission> permissions = permissionsManager.getAllPermissions(group);
        Collections.sort(permissions, (p1, p2) -> {
            return p1.getUser().getDisplayName().compareTo(p2.getUser().getDisplayName());
        });
        return new PaginatedData<>(permissions, request.getPaginatorPage(), request.getPaginatorPageSize());
    }
}
