/*
 * This file is part of gms
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.persistence;

import it.inaf.ia2.gms.DataSourceConfig;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import java.util.ArrayList;
import java.util.Arrays;
import javax.sql.DataSource;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringRunner;

@RunWith(SpringRunner.class)
@ContextConfiguration(classes = DataSourceConfig.class)
public class MembershipsDAOTest {

    private static final String USER_1 = "user-1";
    private static final String USER_2 = "user-2";

    @Autowired
    private DataSource dataSource;

    private GroupsDAO groupsDAO;
    private MembershipsDAO membershipsDAO;

    @Before
    public void setUp() {
        groupsDAO = new GroupsDAO(dataSource);
        membershipsDAO = new MembershipsDAO(dataSource);
    }

    @Test
    public void testAddAndRemoveMembers() {
        groupsDAO.createGroup(groupEntity("A", "A"));
        groupsDAO.createGroup(groupEntity("B", "B"));

        assertTrue(membershipsDAO.findByGroup("A").isEmpty());

        membershipsDAO.addMember(membershipEntity(USER_1, "A"));
        membershipsDAO.addMember(membershipEntity(USER_1, "B"));
        membershipsDAO.addMember(membershipEntity(USER_2, "B"));

        assertTrue(membershipsDAO.findByGroup("A").contains(membershipEntity(USER_1, "A")));
        assertEquals(3, membershipsDAO.findByGroups(Arrays.asList("A", "B")).size());
        assertEquals(2, membershipsDAO.getUserMemberships(USER_1).size());
        assertEquals(1, membershipsDAO.getUserMemberships(USER_2, "").size());

        assertTrue(membershipsDAO.isMemberOf(USER_1, "B"));
        membershipsDAO.removeMembership("B", USER_1);
        assertFalse(membershipsDAO.isMemberOf(USER_1, "B"));

        assertTrue(membershipsDAO.isMemberOf(USER_1, "A"));
        assertTrue(membershipsDAO.isMemberOf(USER_2, "B"));
        membershipsDAO.deleteAllGroupsMembership(Arrays.asList("A", "B"));
        assertFalse(membershipsDAO.isMemberOf(USER_1, "A"));
        assertFalse(membershipsDAO.isMemberOf(USER_2, "B"));
    }

    @Test
    public void testFindByGroupsEmptyList() {
        assertTrue(membershipsDAO.findByGroups(new ArrayList<>()).isEmpty());
    }

    @Test
    public void testDeleteAllGroupsMembershipEmptyList() {
        membershipsDAO.deleteAllGroupsMembership(new ArrayList<>());
    }

    @Test
    public void testIsMemberOfRecursive() {

        groupsDAO.createGroup(groupEntity("ROOT", ""));

        groupsDAO.createGroup(groupEntity("A", "A"));
        groupsDAO.createGroup(groupEntity("E", "A.E"));
        groupsDAO.createGroup(groupEntity("F", "A.E.F"));

        groupsDAO.createGroup(groupEntity("B", "B"));
        groupsDAO.createGroup(groupEntity("C", "B.C"));
        groupsDAO.createGroup(groupEntity("D", "B.C.D"));

        membershipsDAO.addMember(membershipEntity(USER_1, "F"));
        membershipsDAO.addMember(membershipEntity(USER_1, "C"));

        assertFalse(membershipsDAO.isMemberOf(USER_1, "ROOT"));
        assertFalse(membershipsDAO.isMemberOf(USER_1, "A"));
        assertFalse(membershipsDAO.isMemberOf(USER_1, "B"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "C"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "D"));
        assertFalse(membershipsDAO.isMemberOf(USER_1, "E"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "F"));

        membershipsDAO.addMember(membershipEntity(USER_1, "ROOT"));

        assertTrue(membershipsDAO.isMemberOf(USER_1, "ROOT"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "A"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "B"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "C"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "D"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "E"));
        assertTrue(membershipsDAO.isMemberOf(USER_1, "F"));

        assertFalse(membershipsDAO.isMemberOf(USER_1, "inexistent-id"));
    }

    private GroupEntity groupEntity(String groupId, String groupPath) {
        GroupEntity groupEntity = new GroupEntity();
        groupEntity.setId(groupId);
        groupEntity.setName(groupId);
        groupEntity.setPath(groupPath);
        groupEntity.setLeaf(false);
        return groupEntity;
    }

    private MembershipEntity membershipEntity(String userId, String groupId) {
        MembershipEntity membershipEntity = new MembershipEntity();
        membershipEntity.setUserId(userId);
        membershipEntity.setGroupId(groupId);
        return membershipEntity;
    }
}
