/*
 * This file is part of gms
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.manager;

import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.PermissionUtils;
import it.inaf.ia2.gms.authn.RapClient;
import static it.inaf.ia2.gms.persistence.model.ActionType.*;
import it.inaf.ia2.rap.data.RapUser;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class MembershipManager extends UserAwareComponent {

    @Autowired
    private MembershipsDAO membershipsDAO;

    @Autowired
    private GroupsDAO groupsDAO;

    @Autowired
    private PermissionsManager permissionsManager;

    @Autowired
    private RapClient rapClient;

    @Autowired
    private LoggingDAO loggingDAO;

    public List<GroupEntity> getCurrentUserMemberships() {
        return membershipsDAO.getUserMemberships(getCurrentUserId());
    }

    public boolean isCurrentUserMemberOf(String groupId) {
        return membershipsDAO.isMemberOf(getCurrentUserId(), groupId);
    }

    public List<RapUser> getMembers(GroupEntity group) {

        Permission groupPermission = permissionsManager.getCurrentUserPermission(group);

        if (!Permission.includes(groupPermission, Permission.VIEW_MEMBERS)) {
            loggingDAO.logAction(UNAUTHORIZED_ACCESS_ATTEMPT, "Attempted to view members of group " + group.getId());
            throw new UnauthorizedException("You don't have the permission to view members");
        }

        List<MembershipEntity> memberships = membershipsDAO.findByGroup(group.getId());

        Set<String> userIdentifiers = memberships.stream()
                .map(m -> m.getUserId())
                .collect(Collectors.toSet());

        return rapClient.getUsers(userIdentifiers);
    }

    public List<GroupEntity> getUserGroups(GroupEntity parent, String userId) {

        List<PermissionEntity> permissions = permissionsManager.getCurrentUserPermissions(parent);

        List<GroupEntity> allGroups = membershipsDAO.getUserMemberships(userId, parent.getPath());

        // Select only groups visible to the current user
        Set<String> visibleGroupIds = new HashSet<>();
        for (GroupEntity group : allGroups) {
            PermissionUtils.getGroupPermission(group, permissions)
                    .ifPresent(p -> visibleGroupIds.add(group.getId()));
        }
        return groupsDAO.findGroupsByIds(visibleGroupIds);
    }

    public MembershipEntity addMember(GroupEntity group, String userId) {

        verifyUserCanManageMembers(group);

        MembershipEntity membership = new MembershipEntity();
        membership.setGroupId(group.getId());
        membership.setUserId(userId);
        membership.setCreatedBy(getCurrentUserId());

        membership = membershipsDAO.addMember(membership);
        loggingDAO.logAction(MEMBER_ADDED, "Added member, group_id=" + group.getId() + ", user_id=" + userId);

        return membership;
    }

    public void removeMember(GroupEntity group, String userId) {
        verifyUserCanManageMembers(group);
        membershipsDAO.removeMembership(group.getId(), userId);
        loggingDAO.logAction(MEMBER_REMOVED, "Member removed, group_id=" + group.getId() + ", user_id=" + userId);
    }

    private Permission verifyUserCanManageMembers(GroupEntity group) {
        Permission permission = permissionsManager.getCurrentUserPermission(group);
        if (!Permission.includes(permission, Permission.MANAGE_MEMBERS)) {
            loggingDAO.logAction(UNAUTHORIZED_ACCESS_ATTEMPT, "Attempted to manage members of group " + group.getId());
            throw new UnauthorizedException("Missing manage members permissions");
        }
        return permission;
    }
}
