/*
 * This file is part of gms
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.manager.InvitedRegistrationManager;
import it.inaf.ia2.gms.persistence.model.InvitedRegistration;
import it.inaf.ia2.gms.service.GroupNameService;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.List;
import java.util.Scanner;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.support.ServletUriComponentsBuilder;

@Controller
public class InvitedRegistrationController {

    @Autowired
    private InvitedRegistrationManager invitedRegistrationManager;

    @Autowired
    private GroupNameService groupNameService;

    @GetMapping(value = "/invited-registration", produces = MediaType.TEXT_HTML_VALUE)
    public void index(@RequestParam("token") String token, HttpServletRequest request, HttpServletResponse response) throws IOException {
        InvitedRegistration invitedRegistration = invitedRegistrationManager.getInvitedRegistrationFromToken(token);

        String html = getFileContent("invited-registration.html")
                .replace("#EMAIL#", invitedRegistration.getEmail())
                .replace("#GROUPS#", getGroupsList(Collections.singletonList(invitedRegistration)))
                .replace("#HOME#", request.getContextPath());

        response.getOutputStream().print(html);
    }

    @GetMapping(value = "/registration-completed", produces = MediaType.TEXT_HTML_VALUE)
    public void completed(HttpServletRequest request, HttpServletResponse response) throws IOException {

        response.setContentType("text/html;charset=UTF-8");

        List<InvitedRegistration> invitedRegistrations = (List<InvitedRegistration>) request.getAttribute("invited-registrations");
        if (invitedRegistrations == null) {
            // redirect to home
            String baseUrl = ServletUriComponentsBuilder.fromCurrentContextPath().build().toUriString();
            response.sendRedirect(baseUrl);
        } else {
            String html = getFileContent("registration-completed.html")
                    .replace("#GROUPS#", getGroupsList(invitedRegistrations))
                    .replace("#HOME#", request.getContextPath());

            response.getOutputStream().print(html);
        }
    }

    @DeleteMapping(value = "/ui/registration", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<?> deleteInvitedRegistration(@RequestParam("request_id") String requestId, @RequestParam("group_id") String groupId) {
        invitedRegistrationManager.deleteInvitedRegistration(requestId, groupId);
        return ResponseEntity.noContent().build();
    }

    private String getFileContent(String templateFileName) throws IOException {
        try ( InputStream in = InvitedRegistrationController.class.getClassLoader().getResourceAsStream("templates/" + templateFileName)) {
            Scanner s = new Scanner(in).useDelimiter("\\A");
            return s.hasNext() ? s.next() : "";
        }
    }

    private String getGroupsList(List<InvitedRegistration> invitedRegistrations) {
        String groups = "<ul>";
        for (InvitedRegistration invitedRegistration : invitedRegistrations) {
            for (String groupName : groupNameService.getGroupsNamesFromIdentifiers(
                    invitedRegistration.getGroupsPermissions().keySet())) {
                groups += "<li>" + groupName + "</li>";
            }
        }
        groups += "</ul>";
        return groups;
    }
}
