/*
 * This file is part of gms
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.authn.SessionData;
import it.inaf.ia2.gms.manager.InvitedRegistrationManager;
import it.inaf.ia2.gms.model.request.GroupsRequest;
import it.inaf.ia2.gms.model.response.GroupsTabResponse;
import it.inaf.ia2.gms.model.response.HomePageResponse;
import it.inaf.ia2.gms.persistence.model.InvitedRegistration;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.support.ServletUriComponentsBuilder;

@Controller
public class HomePageController {

    @Autowired
    private SessionData session;

    @Autowired
    private GroupsTabResponseBuilder groupsTabResponseBuilder;

    @Autowired
    private InvitedRegistrationManager invitedRegistrationManager;

    @ResponseBody
    @GetMapping(value = "/ui/home", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<HomePageResponse> getMainPage(@Valid GroupsRequest request) {

        HomePageResponse response = new HomePageResponse();

        response.setUser(session.getUserName());

        GroupsTabResponse groupsTabResponse = groupsTabResponseBuilder.getGroupsTab(request);
        response.setBreadcrumbs(groupsTabResponse.getBreadcrumbs());
        response.setGroupsPanel(groupsTabResponse.getGroupsPanel());
        response.setPermission(groupsTabResponse.getPermission());

        return ResponseEntity.ok(response);
    }

    @GetMapping(value = "/", produces = MediaType.TEXT_HTML_VALUE)
    public String index(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

        // This page MUST NOT be cached to avoid losing the login redirect
        response.setHeader("Cache-Control", "no-store, must-revalidate");
        response.setHeader("Expires", "0");

        Optional<List<InvitedRegistration>> optReg = invitedRegistrationManager.completeInvitedRegistrationIfNecessary();
        if (optReg.isPresent()) {
            request.setAttribute("invited-registrations", optReg.get());
            return "/registration-completed";
        }

        return "index.html";
    }

    @GetMapping(value = "/logout", produces = MediaType.TEXT_HTML_VALUE)
    public void logout(HttpSession httpSession, HttpServletResponse response) throws IOException {
        httpSession.invalidate();
        String baseUrl = ServletUriComponentsBuilder.fromCurrentContextPath().build().toUriString();
        response.sendRedirect(baseUrl + "/");
    }
}
