/*
 * This file is part of gms-client
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.gms.client.call;

import it.inaf.ia2.gms.client.BaseGmsClientTest;
import it.inaf.ia2.gms.client.model.Permission;
import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.AdditionalMatchers;
import org.mockito.ArgumentMatchers;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.jupiter.MockitoExtension;

@ExtendWith(MockitoExtension.class)
public class InvitedRegistrationTest extends BaseGmsClientTest {

    @BeforeEach
    @Override
    public void init() {
        super.init();
    }

    @Test
    public void testInvitedRegistration() {

        CompletableFuture response = CompletableFuture.completedFuture(getMockedResponse(201));

        when(httpClient.sendAsync(any(), any())).thenReturn(response);
        Map<String, Permission> permissionsMap = new HashMap<>();
        permissionsMap.put("group1", Permission.MANAGE_MEMBERS);
        permissionsMap.put("group2", Permission.MANAGE_MEMBERS);
        gmsClient.addInvitedRegistration("bvjsgqu423", "email", permissionsMap);
        // hash = AOyojiwaRR7BHPde6Tomg3+BMoQQggNM3wUHEarXuNQ=

        verify(httpClient, times(1)).sendAsync(
                AdditionalMatchers.and(
                        endpointEq("POST", "invited-registration"),
                        ArgumentMatchers.argThat(req -> {
                            String reqbody = req.bodyPublisher().map(p -> {
                                var bodySubscriber = HttpResponse.BodySubscribers.ofString(StandardCharsets.UTF_8);
                                var flowSubscriber = new StringSubscriber(bodySubscriber);
                                p.subscribe(flowSubscriber);
                                return bodySubscriber.getBody().toCompletableFuture().join();
                            }).get();

                            // If the resulting hash contains a + symbol it has to be encoded to %2B,
                            // otherwise it will be interpreted as a space and wrong value will be
                            // stored into the database
                            String expectedBody = "token_hash=AOyojiwaRR7BHPde6Tomg3%2BBMoQQggNM3wUHEarXuNQ="
                                    + "&email=email&groups=group2 MANAGE_MEMBERS\n"
                                    + "group1 MANAGE_MEMBERS";

                            return reqbody.equals(expectedBody);
                        })), any());
    }

    /**
     * Credit: https://stackoverflow.com/a/55816685/771431
     */
    static final class StringSubscriber implements Flow.Subscriber<ByteBuffer> {

        final HttpResponse.BodySubscriber<String> wrapped;

        StringSubscriber(HttpResponse.BodySubscriber<String> wrapped) {
            this.wrapped = wrapped;
        }

        @Override
        public void onSubscribe(Flow.Subscription subscription) {
            wrapped.onSubscribe(subscription);
        }

        @Override
        public void onNext(ByteBuffer item) {
            wrapped.onNext(List.of(item));
        }

        @Override
        public void onError(Throwable throwable) {
            wrapped.onError(throwable);
        }

        @Override
        public void onComplete() {
            wrapped.onComplete();
        }
    }
}
