/* Vuex store, for centralized state management */

import Vue from 'vue';
import Vuex from 'vuex';
import client from 'api-client';

Vue.use(Vuex);

export default new Vuex.Store({
  state: {
    // values populated from API calls
    model: {
      breadcrumbs: [],
      groupsPanel: null,
      permissionsPanel: null,
      membersPanel: null,
      permission: null,
      leaf: false,
      user: null,
      genericSearchResults: [],
      userSearchResults: {
        userLabel: null,
        groups: {},
        permissions: {}
      }
    },
    // values used to perform API calls
    input: {
      selectedGroupId: 'ROOT',
      paginatorPageSize: 20,
      paginatorPage: 1,
      tabIndex: 0,
      searchFilter: '',
      genericSearch: {
        filter: '',
        paginatorPage: 1,
        paginatorPageSize: 20
      }
    },
    loading: false,
    previousPage: null,
    page: 'main'
  },
  mutations: {
    updateHomePageModel(state, model) {
      this.state.model.breadcrumbs = model.breadcrumbs;
      this.state.model.groupsPanel = model.groupsPanel;
      this.state.model.permission = model.permission;
      this.state.model.user = model.user;
    },
    openGroup(state, groupId) {
      let input = this.state.input;
      input.selectedGroupId = groupId;
      input.searchFilter = '';
      client.fetchGroupsTab(input)
        .then(model => {
          this.commit('updateGroups', model);
          if (model.leaf) {
            // If there are no subgroups show the members panel
            this.commit('setTabIndex', 1);
          } else {
            this.commit('setTabIndex', 0);
          }
          this.commit('showMainPage');
        });
    },
    updateGroups(state, model) {
      this.state.model.breadcrumbs = model.breadcrumbs;
      this.state.model.groupsPanel = model.groupsPanel;
      this.state.model.permission = model.permission;
      this.state.model.leaf = model.leaf;
    },
    updateGroupsPanel(state, groupsPanel) {
      this.state.model.groupsPanel = groupsPanel;
      this.state.input.paginatorPage = groupsPanel.currentPage;
    },
    updatePermissionsPanel(state, permissionsPanel) {
      this.state.model.permissionsPanel = permissionsPanel;
      this.state.input.paginatorPage = permissionsPanel.currentPage;
    },
    updateMembersPanel(state, membersPanel) {
      this.state.model.membersPanel = membersPanel;
      this.state.input.paginatorPage = membersPanel.currentPage;
    },
    setTabIndex(state, tabIndex) {
      // this will trigger the tabChanged() method in Main.vue
      this.state.input.tabIndex = tabIndex;
    },
    setLoading(state, loading) {
      this.state.loading = loading;
    },
    showMainPage(state) {
      this.state.page = 'main';
    },
    displaySearchResults(state, results) {
      this.state.page = 'search';
      if (results) {
        this.state.model.genericSearchResults = results;
      }
    },
    updateSearchResults(state, results) {
      this.state.model.genericSearchResults = results;
    },
    backFromUserPage(state) {
      state.page = state.previousPage;
    }
  },
  actions: {
    openUserPage({ state }, userId) {
      state.previousPage = state.page;
      client.openUserSearchResult(userId)
        .then(model => {
          state.page = 'userSearch';
          state.model.userSearchResults.user = model.user;
          state.model.userSearchResults.groups = model.groups;
          state.model.userSearchResults.permissions = model.permissions;
        });
    }
  },
  getters: {
    selectedGroupId: state => {
      return state.model.breadcrumbs[state.model.breadcrumbs.length - 1].groupId;
    }
  }
});
