package it.inaf.ia2.gms.authn;

import it.inaf.ia2.aa.AuthConfig;
import it.inaf.ia2.aa.UserManager;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.core.Ordered;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.config.annotation.CorsRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

@Configuration
public class SecurityConfig {

    private static final Logger LOG = LoggerFactory.getLogger(SecurityConfig.class);

    @Value("${cors.allowed.origin}")
    private String corsAllowedOrigin;

    /**
     * CORS are necessary only for development (API access from npm server).
     */
    @Bean
    @Profile("dev")
    public WebMvcConfigurer corsConfigurer() {

        return new WebMvcConfigurer() {

            @Override
            public void addCorsMappings(CorsRegistry registry) {

                LOG.warn("Development profile active: CORS filter enabled");

                registry.addMapping("/**")
                        .allowedOrigins(corsAllowedOrigin)
                        .allowedMethods("*")
                        .allowCredentials(true);
            }
        };
    }

    @Bean
    public FilterRegistrationBean clientDbFilter(AuthConfig authConfig, UserManager userManager) {
        FilterRegistrationBean bean = new FilterRegistrationBean();
        bean.setFilter(new ClientDbFilter(authConfig, userManager));
        bean.addUrlPatterns("/*");
        bean.setOrder(Ordered.HIGHEST_PRECEDENCE);
        return bean;
    }

    /**
     * Checks JWT for web services.
     */
    @Bean
    public FilterRegistrationBean serviceJWTFilter(LoggingDAO loggingDAO, UserManager userManager) {
        FilterRegistrationBean bean = new FilterRegistrationBean();
        bean.setFilter(new JWTFilter(loggingDAO, userManager));
        bean.addUrlPatterns("/*");
        bean.setOrder(Ordered.HIGHEST_PRECEDENCE);
        return bean;
    }

    @Bean
    public FilterRegistrationBean loginFilterRegistration() {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(new GmsLoginFilter());
        registration.addUrlPatterns("/*");
        return registration;
    }

    @Bean
    public RestTemplate rapRestTemplate() {
        return new RestTemplate();
    }
}
