package it.inaf.ia2.gms.authn;

import static it.inaf.ia2.gms.authn.ClientDbFilter.CLIENT_DB;
import it.inaf.ia2.gms.exception.BadRequestException;
import it.inaf.ia2.rap.client.call.GetUserCall;
import it.inaf.ia2.rap.data.RapUser;
import java.net.URI;
import java.net.http.HttpRequest;
import java.util.List;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientDbRapClient extends ServletRapClient {

    private static final Logger LOG = LoggerFactory.getLogger(ClientDbRapClient.class);

    public ClientDbRapClient(String baseUrl) {
        super(baseUrl);
    }

    @Override
    protected HttpRequest.Builder newAuthRequest(HttpRequest.Builder requestBuilder, HttpServletRequest request) {
        return setClientDb(super.newClientSecretRequest(requestBuilder), request);
    }

    @Override
    public HttpRequest.Builder newRequest(String endpoint, HttpServletRequest context) {
        return setClientDb(super.newRequest(endpoint), context);
    }

    @Override
    public HttpRequest.Builder newRequest(URI uri, HttpServletRequest context) {
        return setClientDb(super.newRequest(uri), context);
    }

    private HttpRequest.Builder setClientDb(HttpRequest.Builder builder, HttpServletRequest request) {
        HttpSession session = request.getSession(false);
        if (session != null) {
            String clientDb = (String) session.getAttribute("client_db");
            if (clientDb != null) {
                builder.setHeader("client_db", clientDb);
                LOG.debug("client_db=" + clientDb);
            }
        }
        return builder;
    }

    @Override
    public URI getAuthorizationUri(HttpServletRequest request) {
        // for a better security we should check for allowed redirects
        String redirect = request.getParameter("redirect");

        URI uri;
        if (redirect != null) {
            uri = URI.create(redirect);
        } else {
            uri = super.getAuthorizationUri(request);
        }

        String clientDb = request.getParameter(CLIENT_DB);
        if (clientDb == null) {
            HttpSession session = request.getSession(false);
            if (session != null) {
                clientDb = (String) session.getAttribute(CLIENT_DB);
            }
        }
        if (clientDb == null) {
            throw new BadRequestException("client_db not set");
        }

        redirect = uri.toString();

        redirect += redirect.contains("?") ? "&" : "?";
        redirect += CLIENT_DB + "=" + clientDb;

        return URI.create(redirect);
    }

    @Override
    public URI getAccessTokenUri(HttpServletRequest request) {
        String tokenUri = request.getParameter("token_uri");
        if (tokenUri != null) {
            return URI.create(tokenUri);
        }
        return super.getAccessTokenUri(request);
    }

    @Override
    public List<RapUser> getUsers(String searchText, HttpServletRequest request) {
        List<RapUser> users = new GetUserCall(this).getUsers(searchText, request);
        return users.stream()
                .filter(u -> u.getDisplayName().contains(searchText) || u.getPrimaryEmailAddress().contains(searchText))
                .collect(Collectors.toList());
    }
}
