package it.inaf.ia2.gms.service;

import static it.inaf.ia2.gms.controller.ControllersMockData.*;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.persistence.GroupsDAO;
import it.inaf.ia2.gms.persistence.InvitedRegistrationDAO;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import org.junit.Test;
import org.junit.runner.RunWith;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class GroupsServiceTest {
    
    @Mock
    private GroupsDAO groupsDAO;
    @Mock
    private PermissionsDAO permissionsDAO;
    @Mock
    private MembershipsDAO membershipsDAO;
    @Mock
    private InvitedRegistrationDAO invitedRegistrationDAO;
    @Mock
    private LoggingDAO loggingDAO;
    
    @InjectMocks
    private GroupsService groupsService;
    
    private final GroupEntity root = group("ROOT", "ROOT", "");
    
    @Test
    public void testAddGroup() {
        
        GroupEntity group1 = groupsService.addGroup(root, "group1", false, "creator_id");
        
        verify(groupsDAO, times(1)).createGroup(argThat(g -> "group1".equals(g.getName())));
        
        assertNotNull(group1.getId());
        assertEquals("group1", group1.getName());
        assertFalse(group1.isLeaf());
        assertEquals("creator_id", group1.getCreatedBy());
        assertNotNull(group1.getCreationTime());
        
        GroupEntity group2 = groupsService.addGroup(group1, "group2", true, "creator_id");
        
        verify(groupsDAO, times(1)).createGroup(argThat(g -> "group2".equals(g.getName())));
        
        assertNotNull(group2.getId());
        assertEquals("group2", group2.getName());
        assertTrue(group2.isLeaf());
        assertEquals(group1.getId() + "." + group2.getId(), group2.getPath());
    }
    
    @Test
    public void testGetChildGroups() {
        groupsService.getChildGroups(getLbtGroup(), false);
        verify(groupsDAO, times(1)).getDirectSubGroups(getLbtGroup().getPath());
    }
    
    @Test
    public void testGetChildGroupsRecursive() {
        groupsService.getChildGroups(getLbtGroup(), true);
        verify(groupsDAO, times(1)).getAllChildren(getLbtGroup().getPath());
    }
    
    @Test
    public void testDeleteRootIsDenied() {
        boolean exception = false;
        try {
            groupsService.deleteGroup(root);
        } catch (UnauthorizedException ex) {
            exception = true;
        }
        assertTrue(exception);
    }
    
    @Test
    public void testDeleteLockedGroupIsDenied() {
        GroupEntity group = getLbtGroup();
        group.setLocked(true);
        boolean exception = false;
        try {
            groupsService.deleteGroup(group);
        } catch (UnauthorizedException ex) {
            exception = true;
        }
        assertTrue(exception);
    }
    
    @Test
    public void testDeleteGroup() {
        
        GroupEntity lbt = getLbtGroup();
        GroupEntity inaf = getInafGroup();
        
        when(groupsDAO.findGroupByPath(lbt.getPath())).thenReturn(Optional.of(lbt));
        when(groupsDAO.getAllChildren(inaf.getPath())).thenReturn(new ArrayList<>(List.of(getInafProgramGroup())));
        
        groupsService.deleteGroup(inaf);
        
        verify(groupsDAO, times(2)).deleteGroup(any());
    }
    
    private GroupEntity group(String id, String name, String path) {
        GroupEntity group = new GroupEntity();
        group.setId(id);
        group.setName(name);
        group.setPath(path);
        return group;
    }
}
