package it.inaf.ia2.gms.service;

import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.JoinDAO;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.MembershipsDAO;
import it.inaf.ia2.gms.persistence.PermissionsDAO;
import it.inaf.ia2.gms.persistence.model.ActionType;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.MembershipEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class JoinService {

    @Autowired
    private MembershipsDAO membershipsDAO;

    @Autowired
    private PermissionsDAO permissionsDAO;

    @Autowired
    private JoinDAO joinDAO;

    @Autowired
    private LoggingDAO loggingDAO;

    public String join(String userId1, String userId2) {

        List<GroupEntity> user1Groups = membershipsDAO.getUserMemberships(userId1);
        List<GroupEntity> user2Groups = membershipsDAO.getUserMemberships(userId2);

        // The user having less groups will be deleted
        String remainingUserId = user1Groups.size() >= user2Groups.size() ? userId1 : userId2;
        String deletingUserId = remainingUserId.equals(userId1) ? userId2 : userId1;

        List<GroupEntity> remainingUserGroups = remainingUserId.equals(userId1) ? user1Groups : user2Groups;
        List<GroupEntity> deletingUserGroups = deletingUserId.equals(userId1) ? user1Groups : user2Groups;

        Set<MembershipEntity> existingMemberships = remainingUserGroups.stream()
                .map(g -> getMembershipEntity(g.getId(), remainingUserId))
                .collect(Collectors.toSet());

        Set<MembershipEntity> membershipsToAdd = deletingUserGroups.stream()
                .map(g -> getMembershipEntity(g.getId(), remainingUserId))
                .filter(m -> !existingMemberships.contains(m))
                .collect(Collectors.toSet());

        Set<PermissionEntity> existingPermissions
                = permissionsDAO.findUserPermissions(remainingUserId).stream()
                        .collect(Collectors.toSet());

        Set<PermissionEntity> permissionsToAdd
                = permissionsDAO.findUserPermissions(deletingUserId).stream()
                        .map(p -> {
                            p.setUserId(remainingUserId);
                            return p;
                        })
                        .filter(p -> isPermissionToAdd(existingPermissions, p))
                        .collect(Collectors.toSet());

        joinDAO.join(membershipsToAdd, permissionsToAdd, deletingUserId);

        loggingDAO.logAction(ActionType.JOIN, "removed_user=" + deletingUserId
                + " added_memberships: " + String.join(", ", getAddedGroups(membershipsToAdd))
                + " added_permissions: " + String.join(", ", getAddedPermissions(permissionsToAdd))
        );

        return remainingUserId;
    }

    private List<String> getAddedGroups(Set<MembershipEntity> membershipsToAdd) {
        return membershipsToAdd.stream().map(m -> m.getGroupId()).collect(Collectors.toList());
    }

    private List<String> getAddedPermissions(Set<PermissionEntity> permissionsToAdd) {
        return permissionsToAdd.stream().map(m -> "(" + m.getGroupId() + "," + m.getPermission() + ")").collect(Collectors.toList());
    }

    private MembershipEntity getMembershipEntity(String groupId, String userId) {
        MembershipEntity entity = new MembershipEntity();
        entity.setGroupId(groupId);
        entity.setUserId(userId);
        return entity;
    }

    private boolean isPermissionToAdd(Set<PermissionEntity> existingPermissions, PermissionEntity permissionToCheck) {
        for (PermissionEntity permission : existingPermissions) {
            if (permission.getGroupId().equals(permissionToCheck.getGroupId())
                    && permission.getUserId().equals(permissionToCheck.getUserId())) {
                if (permission.getPermission() == permissionToCheck.getPermission()) {
                    return false;
                }
                Permission strongerPermission = Permission.addPermission(
                        permission.getPermission(), permissionToCheck.getPermission());
                return permission.getPermission() != strongerPermission;
            }
        }
        return true;
    }
}
