package it.inaf.ia2.gms.manager;

import it.inaf.ia2.gms.exception.BadRequestException;
import it.inaf.ia2.gms.exception.UnauthorizedException;
import it.inaf.ia2.gms.model.Permission;
import it.inaf.ia2.gms.persistence.LoggingDAO;
import it.inaf.ia2.gms.persistence.model.ActionType;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.persistence.model.PermissionEntity;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.gms.service.PermissionUtils;
import java.util.ArrayList;
import java.util.List;
import org.springframework.stereotype.Service;

@Service
public class GroupsManager extends UserAwareComponent {

    private final GroupsService groupsService;
    private final PermissionsManager permissionsManager;
    private final LoggingDAO loggingDAO;

    public GroupsManager(GroupsService groupsService, PermissionsManager permissionsManager, LoggingDAO loggingDAO) {
        this.groupsService = groupsService;
        this.permissionsManager = permissionsManager;
        this.loggingDAO = loggingDAO;
    }

    public GroupEntity getRoot() {
        return getGroupById(GroupsService.ROOT);
    }

    public GroupEntity getGroupById(String groupId) {
        GroupEntity group = groupsService.getGroupById(groupId);
        verifyUserCanReadGroup(group);
        return group;
    }

    public GroupEntity getGroupByPath(String path) {
        GroupEntity group = groupsService.getGroupByPath(path);
        verifyUserCanReadGroup(group);
        return group;
    }

    public List<GroupEntity> getChildGroups(GroupEntity parentGroup, boolean recursive) {
        List<GroupEntity> allSubGroups = groupsService.getChildGroups(parentGroup, recursive);

        // Select only the groups visible to the user
        List<PermissionEntity> permissions = permissionsManager.getCurrentUserPermissions();
        List<GroupEntity> visibleSubgroups = new ArrayList<>();

        for (GroupEntity subgroup : allSubGroups) {
            PermissionUtils.getGroupPermission(subgroup, permissions).ifPresent(permission -> {
                visibleSubgroups.add(subgroup);
            });
        }

        return visibleSubgroups;
    }

    public GroupEntity createGroup(GroupEntity parent, String childGroupName, boolean leaf) {

        if (parent.isLeaf()) {
            throw new BadRequestException("Unable to create a sub group inside a leaf group");
        }

        verifyUserCanManageGroup(parent);

        return groupsService.addGroup(parent, childGroupName, leaf, getCurrentUserId());
    }

    public GroupEntity updateGroup(String groupId, String newGroupName, boolean leaf) {

        GroupEntity group = groupsService.getGroupById(groupId);
        verifyUserCanManageGroup(group);

        return groupsService.updateGroup(group, newGroupName, leaf);
    }

    public GroupEntity deleteGroup(String groupId) {
        GroupEntity group = groupsService.getGroupById(groupId);
        verifyUserCanManageGroup(group);
        return groupsService.deleteGroup(group);
    }

    public void verifyUserCanReadGroup(GroupEntity group) {
        if (GroupsService.ROOT.equals(group.getId())) {
            // Everybody can read the root
            return;
        }
        if (permissionsManager.getCurrentUserPermission(group) == null) {
            loggingDAO.logAction(ActionType.UNAUTHORIZED_ACCESS_ATTEMPT, "Unauthorized group management request, group_id=" + group.getId());
            throw new UnauthorizedException("Missing permission to see this group");
        }
    }

    private void verifyUserCanManageGroup(GroupEntity group) {
        if (permissionsManager.getCurrentUserPermission(group) != Permission.ADMIN) {
            loggingDAO.logAction(ActionType.UNAUTHORIZED_ACCESS_ATTEMPT, "Unauthorized group management request, group_id=" + group.getId());
            throw new UnauthorizedException("Missing admin permission");
        }
    }
}
