package it.inaf.ia2.gms.controller;

import it.inaf.ia2.gms.manager.MembershipManager;
import it.inaf.ia2.gms.manager.PermissionsManager;
import it.inaf.ia2.gms.model.request.AddMemberRequest;
import it.inaf.ia2.gms.model.response.PaginatedData;
import it.inaf.ia2.gms.model.request.PaginatedModelRequest;
import it.inaf.ia2.gms.model.request.RemoveMemberRequest;
import it.inaf.ia2.gms.model.request.TabRequest;
import it.inaf.ia2.gms.persistence.model.GroupEntity;
import it.inaf.ia2.gms.service.GroupNameService;
import it.inaf.ia2.gms.service.GroupsService;
import it.inaf.ia2.rap.data.RapUser;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class MembersController {

    @Autowired
    private GroupsService groupsService;

    @Autowired
    private MembershipManager membershipManager;

    @Autowired
    private PermissionsManager permissionsManager;

    @Autowired
    protected GroupNameService groupNameService;

    @GetMapping(value = "/ui/members", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> getMembersTab(TabRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        PaginatedData<RapUser> membersPanel = getMembersPanel(group, request);

        return ResponseEntity.ok(membersPanel);
    }

    @PostMapping(value = "/ui/member", consumes = MediaType.APPLICATION_JSON_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> addMember(@Valid @RequestBody AddMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        membershipManager.addMember(group, request.getUserId());
        permissionsManager.addPermission(group, request.getUserId(), request.getPermission());

        return new ResponseEntity<>(getMembersPanel(group, request), HttpStatus.CREATED);
    }

    @DeleteMapping(value = "/ui/member", produces = MediaType.APPLICATION_JSON_VALUE)
    public ResponseEntity<PaginatedData<RapUser>> removeMember(@Valid RemoveMemberRequest request) {

        GroupEntity group = groupsService.getGroupById(request.getGroupId());

        membershipManager.removeMember(group, request.getUserId());

        if (request.isRemoveAlsoPermission()) {
            permissionsManager.removePermission(group, request.getUserId());
        }

        return ResponseEntity.ok(getMembersPanel(group, request));
    }

    private PaginatedData<RapUser> getMembersPanel(GroupEntity group, PaginatedModelRequest request) {
        List<RapUser> members = membershipManager.getMembers(group);
        Collections.sort(members, (m1, m2) -> {
            return m1.getDisplayName().compareTo(m2.getDisplayName());
        });
        return new PaginatedData<>(members, request.getPaginatorPage(), request.getPaginatorPageSize());
    }

    @GetMapping(value = "/membership", produces = MediaType.TEXT_PLAIN_VALUE)
    public void getMembership(@RequestParam("group") Optional<String> groupNames, @RequestParam("user_id") String userId, HttpServletResponse response) throws IOException {

        GroupEntity parent = groupNameService.getGroupFromNames(groupNames);

        List<GroupEntity> groups = membershipManager.getUserGroups(parent, userId);

        try ( PrintWriter pw = new PrintWriter(response.getOutputStream())) {
            for (String groupName : groupNameService.getGroupsNames(groups)) {
                pw.println(groupNameService.getShortGroupName(groupName, groupNames));
            }
        }
    }

    @PostMapping(value = "/membership", produces = MediaType.TEXT_PLAIN_VALUE, consumes = MediaType.APPLICATION_FORM_URLENCODED_VALUE)
    public void addMember(@RequestParam("group") Optional<String> groupNames,
            @RequestParam("user_id") String targetUserId) {

        GroupEntity groupEntity = groupNameService.getGroupFromNames(groupNames);

        membershipManager.addMember(groupEntity, targetUserId);
    }

    @DeleteMapping(value = "/membership", produces = MediaType.TEXT_PLAIN_VALUE)
    public ResponseEntity<?> removeMember(@RequestParam("group") Optional<String> groupNames, @RequestParam("user_id") String userId) {

        GroupEntity groupEntity = groupNameService.getGroupFromNames(groupNames);
        membershipManager.removeMember(groupEntity, userId);

        return ResponseEntity.noContent().build();
    }
}
