/* Vuex store, for centralized state management */

import Vue from 'vue';
import Vuex from 'vuex';
import client from 'api-client';
import router from './router.js';

Vue.use(Vuex);

export default new Vuex.Store({
  state: {
    // values populated from API calls
    model: {
      breadcrumbs: [],
      groupsPanel: null,
      permissionsPanel: null,
      membersPanel: null,
      invitedRegistrations: null,
      permission: null,
      leaf: false,
      user: null,
      genericSearchResults: {},
      userSearchResults: {
        user: null,
        groups: {},
        permissions: {}
      }
    },
    // values used to perform API calls
    input: {
      selectedGroupId: 'ROOT',
      paginatorPageSize: 20,
      paginatorPage: 1,
      tabIndex: 0,
      previousTabIndex: null,
      searchFilter: '',
      genericSearch: {
        filter: '',
        paginatorPage: 1,
        paginatorPageSize: 20,
        users: true,
        groups: true
      }
    },
    loading: false
  },
  mutations: {
    updateHomePageModel(state, model) {
      state.model.breadcrumbs = model.breadcrumbs;
      state.model.groupsPanel = model.groupsPanel;
      state.model.permission = model.permission;
      state.model.invitedRegistrations = model.invitedRegistrations;
      state.model.user = model.user;
    },
    updateGroups(state, model) {
      state.model.breadcrumbs = model.breadcrumbs;
      state.model.groupsPanel = model.groupsPanel;
      state.model.permission = model.permission;
      state.model.invitedRegistrations = model.invitedRegistrations;
      state.model.leaf = model.leaf;
    },
    updateGroupsPanel(state, groupsPanel) {
      state.model.groupsPanel = groupsPanel;
      state.input.paginatorPage = groupsPanel.currentPage;
    },
    updatePermissionsPanel(state, permissionsPanel) {
      state.model.permissionsPanel = permissionsPanel;
      for (let up of permissionsPanel.items) {
        Vue.set(up, 'editable', false);
      }
      state.input.paginatorPage = permissionsPanel.currentPage;
    },
    togglePermissionEditable(state, index) {
      let up = state.model.permissionsPanel.items[index];
      up.editable = !up.editable;
    },
    updateMembersPanel(state, membersPanel) {
      state.model.membersPanel = membersPanel;
      state.input.paginatorPage = membersPanel.currentPage;
    },
    setTabIndex(state, tabIndex) {
      // this will trigger the tabChanged() method in Main.vue
      state.input.tabIndex = tabIndex;
    },
    setLoading(state, loading) {
      state.loading = loading;
    },
    displaySearchResults(state, results) {
      if (results) {
        state.model.genericSearchResults = results;
      }
    },
    updateSearchResults(state, results) {
      state.model.genericSearchResults = results;
    },
    setUserSearchModel(state, model) {
      state.model.userSearchResults.user = model.user;
      state.model.userSearchResults.groups = model.groups;
      state.model.userSearchResults.permissions = model.permissions;
    },
    setGenericSearchFilter(state, filter) {
      state.input.genericSearch.filter = filter;
      state.input.genericSearch.paginatorPage = 1;
      state.input.genericSearch.paginatorPageSize = 20;
      state.input.genericSearch.users = true;
      state.input.genericSearch.groups = true;
    },
    removeInvitedRegistration(state, regId) {
      let index = -1;
      for (var i = 0; i < state.model.invitedRegistrations.length; i++) {
        if (state.model.invitedRegistrations[i].id === regId) {
          index = i;
          break;
        }
      }
      if (index !== -1) {
        state.model.invitedRegistrations.splice(index, 1);
      }
    }
  },
  actions: {
    loadHomePageModel({ commit, state }) {
      client.fetchHomePageModel(state.input)
        .then(model => commit('updateHomePageModel', model));
    },
    search({ commit, state }, filter) {
      if (state.input.genericSearch.filter !== filter) {
        commit('setGenericSearchFilter', filter);
      }
      client.search(state.input)
        .then(results => commit('displaySearchResults', results));
    },
    /**
     * tabIndex parameter is equal to state.input.tabIndex when this method
     * is called as a result of the @input event on the b-tabs component.
     * For this reason the additional field state.input.previousTabIndex is used
     * to check if the tab index changed and perform the AJAX calls only when
     * they are really needed.
     */
    changeTab({ commit, state }, tabIndex) {
      let skip = tabIndex === state.input.previousTabIndex;
      state.input.previousTabIndex = tabIndex;
      state.input.tabIndex = tabIndex;
      if (skip) {
        return;
      }
      // reset paginator
      state.input.paginatorPage = 1;
      switch (tabIndex) {
        case 0:
          client.fetchGroupsTab(state.input)
            .then(model => commit('updateGroups', model));
          break;
        case 1:
          client.fetchMembersPanel(state.input)
            .then(panel => commit('updateMembersPanel', panel));
          break;
        case 2:
          client.fetchPermissionsPanel(state.input)
            .then(panel => commit('updatePermissionsPanel', panel));
          break;
      }
    },
    changeBreadcrumb({ state, commit, dispatch }, groupId) {
      state.input.selectedGroupId = groupId;
      state.input.searchFilter = null;
      if (state.input.tabIndex === 0) {
        client.fetchGroupsTab(state.input)
          .then(model => {
            commit('updateGroups', model);
            dispatch('changeTab', 0);
          });
      } else {
        dispatch('changeTab', 0);
      }
    },
    updateCurrentGroup({ dispatch, state }, data) {
      state.model.breadcrumbs[state.model.breadcrumbs.length - 1].groupName = data.newGroupName;
      state.model.leaf = data.leaf;
      if (state.input.tabIndex === 0 && state.model.leaf) {
        dispatch('changeTab', 1);
      }
    },
    openGroup({ commit, dispatch, state }, groupId) {
      let input = state.input;
      input.selectedGroupId = groupId;
      input.searchFilter = '';
      input.paginatorPage = 1;
      input.tabIndex = 0;
      input.previousTabIndex = 0;
      client.fetchGroupsTab(input)
        .then(model => {
          commit('updateGroups', model);
          if (model.leaf) {
            // If there are no subgroups show the members panel
            dispatch('changeTab', 1);
          }
          router.push('/', () => {});
        });
    },
    openUserPage({ commit }, userId) {
      client.openUserSearchResult(userId)
        .then(model => commit('setUserSearchModel', model));
    }
  },
  getters: {
    selectedGroupId: state => {
      return state.model.breadcrumbs[state.model.breadcrumbs.length - 1].groupId;
    }
  }
});
